import { formatDocsLink } from "../terminal/links.js";
import { theme } from "../terminal/theme.js";
import { formatHelpExampleGroup } from "./help-format.js";
import { createDefaultDeps } from "./deps.js";
import { runDaemonInstall, runDaemonRestart, runDaemonStart, runDaemonStatus, runDaemonStop, runDaemonUninstall, } from "./daemon-cli/runners.js";
import { runNodeDaemonInstall, runNodeDaemonRestart, runNodeDaemonStart, runNodeDaemonStatus, runNodeDaemonStop, runNodeDaemonUninstall, } from "./node-cli/daemon.js";
export function registerServiceCli(program) {
    const gatewayExamples = [
        ["clawdbot service gateway status", "Show gateway service status + probe."],
        [
            "clawdbot service gateway install --port 18789 --token <token>",
            "Install the Gateway service on port 18789.",
        ],
        ["clawdbot service gateway restart", "Restart the Gateway service."],
    ];
    const nodeExamples = [
        ["clawdbot service node status", "Show node host service status."],
        [
            "clawdbot service node install --host gateway.local --port 18789 --tls",
            "Install the node host service with TLS.",
        ],
        ["clawdbot service node restart", "Restart the node host service."],
    ];
    const service = program
        .command("service")
        .description("Manage Gateway and node host services (launchd/systemd/schtasks)")
        .addHelpText("after", () => `\n${theme.heading("Examples:")}\n${formatHelpExampleGroup("Gateway:", gatewayExamples)}\n\n${formatHelpExampleGroup("Node:", nodeExamples)}\n\n${theme.muted("Docs:")} ${formatDocsLink("/cli/service", "docs.clawd.bot/cli/service")}\n`);
    const gateway = service.command("gateway").description("Manage the Gateway service");
    gateway
        .command("status")
        .description("Show gateway service status + probe the Gateway")
        .option("--url <url>", "Gateway WebSocket URL (defaults to config/remote/local)")
        .option("--token <token>", "Gateway token (if required)")
        .option("--password <password>", "Gateway password (password auth)")
        .option("--timeout <ms>", "Timeout in ms", "10000")
        .option("--no-probe", "Skip RPC probe")
        .option("--deep", "Scan system-level services", false)
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runDaemonStatus({
            rpc: opts,
            probe: Boolean(opts.probe),
            deep: Boolean(opts.deep),
            json: Boolean(opts.json),
        });
    });
    gateway
        .command("install")
        .description("Install the Gateway service (launchd/systemd/schtasks)")
        .option("--port <port>", "Gateway port")
        .option("--runtime <runtime>", "Service runtime (node|bun). Default: node")
        .option("--token <token>", "Gateway token (token auth)")
        .option("--force", "Reinstall/overwrite if already installed", false)
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runDaemonInstall(opts);
    });
    gateway
        .command("uninstall")
        .description("Uninstall the Gateway service (launchd/systemd/schtasks)")
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runDaemonUninstall(opts);
    });
    gateway
        .command("start")
        .description("Start the Gateway service (launchd/systemd/schtasks)")
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runDaemonStart(opts);
    });
    gateway
        .command("stop")
        .description("Stop the Gateway service (launchd/systemd/schtasks)")
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runDaemonStop(opts);
    });
    gateway
        .command("restart")
        .description("Restart the Gateway service (launchd/systemd/schtasks)")
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runDaemonRestart(opts);
    });
    const node = service.command("node").description("Manage the node host service");
    node
        .command("status")
        .description("Show node host service status")
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runNodeDaemonStatus(opts);
    });
    node
        .command("install")
        .description("Install the node host service (launchd/systemd/schtasks)")
        .option("--host <host>", "Gateway host")
        .option("--port <port>", "Gateway port")
        .option("--tls", "Use TLS for the Gateway connection", false)
        .option("--tls-fingerprint <sha256>", "Expected TLS certificate fingerprint (sha256)")
        .option("--node-id <id>", "Override node id (clears pairing token)")
        .option("--display-name <name>", "Override node display name")
        .option("--runtime <runtime>", "Service runtime (node|bun). Default: node")
        .option("--force", "Reinstall/overwrite if already installed", false)
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runNodeDaemonInstall(opts);
    });
    node
        .command("uninstall")
        .description("Uninstall the node host service (launchd/systemd/schtasks)")
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runNodeDaemonUninstall(opts);
    });
    node
        .command("start")
        .description("Start the node host service (launchd/systemd/schtasks)")
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runNodeDaemonStart(opts);
    });
    node
        .command("stop")
        .description("Stop the node host service (launchd/systemd/schtasks)")
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runNodeDaemonStop(opts);
    });
    node
        .command("restart")
        .description("Restart the node host service (launchd/systemd/schtasks)")
        .option("--json", "Output JSON", false)
        .action(async (opts) => {
        await runNodeDaemonRestart(opts);
    });
    // Build default deps (parity with daemon CLI).
    void createDefaultDeps();
}
