import { resolveGatewayPort, writeConfigFile } from "../../config/config.js";
import { logConfigUpdated } from "../../config/logging.js";
import { formatCliCommand } from "../../cli/command-format.js";
import { DEFAULT_GATEWAY_DAEMON_RUNTIME } from "../daemon-runtime.js";
import { healthCommand } from "../health.js";
import { applyWizardMetadata, DEFAULT_WORKSPACE, ensureWorkspaceAndSessions, resolveControlUiLinks, waitForGatewayReachable, } from "../onboard-helpers.js";
import { applyNonInteractiveAuthChoice } from "./local/auth-choice.js";
import { installGatewayDaemonNonInteractive } from "./local/daemon-install.js";
import { applyNonInteractiveGatewayConfig } from "./local/gateway-config.js";
import { logNonInteractiveOnboardingJson } from "./local/output.js";
import { applyNonInteractiveSkillsConfig } from "./local/skills-config.js";
import { resolveNonInteractiveWorkspaceDir } from "./local/workspace.js";
export async function runNonInteractiveOnboardingLocal(params) {
    const { opts, runtime, baseConfig } = params;
    const mode = "local";
    const workspaceDir = resolveNonInteractiveWorkspaceDir({
        opts,
        baseConfig,
        defaultWorkspaceDir: DEFAULT_WORKSPACE,
    });
    let nextConfig = {
        ...baseConfig,
        agents: {
            ...baseConfig.agents,
            defaults: {
                ...baseConfig.agents?.defaults,
                workspace: workspaceDir,
            },
        },
        gateway: {
            ...baseConfig.gateway,
            mode: "local",
        },
    };
    const authChoice = opts.authChoice ?? "skip";
    const nextConfigAfterAuth = await applyNonInteractiveAuthChoice({
        nextConfig,
        authChoice,
        opts,
        runtime,
        baseConfig,
    });
    if (!nextConfigAfterAuth)
        return;
    nextConfig = nextConfigAfterAuth;
    const gatewayBasePort = resolveGatewayPort(baseConfig);
    const gatewayResult = applyNonInteractiveGatewayConfig({
        nextConfig,
        opts,
        runtime,
        defaultPort: gatewayBasePort,
    });
    if (!gatewayResult)
        return;
    nextConfig = gatewayResult.nextConfig;
    nextConfig = applyNonInteractiveSkillsConfig({ nextConfig, opts, runtime });
    nextConfig = applyWizardMetadata(nextConfig, { command: "onboard", mode });
    await writeConfigFile(nextConfig);
    logConfigUpdated(runtime);
    await ensureWorkspaceAndSessions(workspaceDir, runtime, {
        skipBootstrap: Boolean(nextConfig.agents?.defaults?.skipBootstrap),
    });
    await installGatewayDaemonNonInteractive({
        nextConfig,
        opts,
        runtime,
        port: gatewayResult.port,
        gatewayToken: gatewayResult.gatewayToken,
    });
    const daemonRuntimeRaw = opts.daemonRuntime ?? DEFAULT_GATEWAY_DAEMON_RUNTIME;
    if (!opts.skipHealth) {
        const links = resolveControlUiLinks({
            bind: gatewayResult.bind,
            port: gatewayResult.port,
            customBindHost: nextConfig.gateway?.customBindHost,
            basePath: undefined,
        });
        await waitForGatewayReachable({
            url: links.wsUrl,
            token: gatewayResult.gatewayToken,
            deadlineMs: 15_000,
        });
        await healthCommand({ json: false, timeoutMs: 10_000 }, runtime);
    }
    logNonInteractiveOnboardingJson({
        opts,
        runtime,
        mode,
        workspaceDir,
        authChoice,
        gateway: {
            port: gatewayResult.port,
            bind: gatewayResult.bind,
            authMode: gatewayResult.authMode,
            tailscaleMode: gatewayResult.tailscaleMode,
        },
        installDaemon: Boolean(opts.installDaemon),
        daemonRuntime: opts.installDaemon ? daemonRuntimeRaw : undefined,
        skipSkills: Boolean(opts.skipSkills),
        skipHealth: Boolean(opts.skipHealth),
    });
    if (!opts.json) {
        runtime.log(`Tip: run \`${formatCliCommand("openclaw configure --section web")}\` to store your Brave API key for web_search. Docs: https://docs.openclaw.ai/tools/web`);
    }
}
