import { execFile } from "node:child_process";
import { X509Certificate } from "node:crypto";
import fs from "node:fs/promises";
import path from "node:path";
import { promisify } from "node:util";
import { CONFIG_DIR, ensureDir, resolveUserPath, shortenHomeInString } from "../../../utils.js";
const execFileAsync = promisify(execFile);
function normalizeFingerprint(input) {
    return input.replace(/[^a-fA-F0-9]/g, "").toLowerCase();
}
async function fileExists(filePath) {
    try {
        await fs.access(filePath);
        return true;
    }
    catch {
        return false;
    }
}
async function generateSelfSignedCert(params) {
    const certDir = path.dirname(params.certPath);
    const keyDir = path.dirname(params.keyPath);
    await ensureDir(certDir);
    if (keyDir !== certDir) {
        await ensureDir(keyDir);
    }
    await execFileAsync("openssl", [
        "req",
        "-x509",
        "-newkey",
        "rsa:2048",
        "-sha256",
        "-days",
        "3650",
        "-nodes",
        "-keyout",
        params.keyPath,
        "-out",
        params.certPath,
        "-subj",
        "/CN=clawdbot-bridge",
    ]);
    await fs.chmod(params.keyPath, 0o600).catch(() => { });
    await fs.chmod(params.certPath, 0o600).catch(() => { });
    params.log?.info?.(`bridge tls: generated self-signed cert at ${shortenHomeInString(params.certPath)}`);
}
export async function loadBridgeTlsRuntime(cfg, log) {
    if (!cfg || cfg.enabled !== true)
        return { enabled: false, required: false };
    const autoGenerate = cfg.autoGenerate !== false;
    const baseDir = path.join(CONFIG_DIR, "bridge", "tls");
    const certPath = resolveUserPath(cfg.certPath ?? path.join(baseDir, "bridge-cert.pem"));
    const keyPath = resolveUserPath(cfg.keyPath ?? path.join(baseDir, "bridge-key.pem"));
    const caPath = cfg.caPath ? resolveUserPath(cfg.caPath) : undefined;
    const hasCert = await fileExists(certPath);
    const hasKey = await fileExists(keyPath);
    if (!hasCert && !hasKey && autoGenerate) {
        try {
            await generateSelfSignedCert({ certPath, keyPath, log });
        }
        catch (err) {
            return {
                enabled: false,
                required: true,
                certPath,
                keyPath,
                error: `bridge tls: failed to generate cert (${String(err)})`,
            };
        }
    }
    if (!(await fileExists(certPath)) || !(await fileExists(keyPath))) {
        return {
            enabled: false,
            required: true,
            certPath,
            keyPath,
            error: "bridge tls: cert/key missing",
        };
    }
    try {
        const cert = await fs.readFile(certPath, "utf8");
        const key = await fs.readFile(keyPath, "utf8");
        const ca = caPath ? await fs.readFile(caPath, "utf8") : undefined;
        const x509 = new X509Certificate(cert);
        const fingerprintSha256 = normalizeFingerprint(x509.fingerprint256 ?? "");
        if (!fingerprintSha256) {
            return {
                enabled: false,
                required: true,
                certPath,
                keyPath,
                caPath,
                error: "bridge tls: unable to compute certificate fingerprint",
            };
        }
        return {
            enabled: true,
            required: true,
            certPath,
            keyPath,
            caPath,
            fingerprintSha256,
            tlsOptions: {
                cert,
                key,
                ca,
                minVersion: "TLSv1.2",
            },
        };
    }
    catch (err) {
        return {
            enabled: false,
            required: true,
            certPath,
            keyPath,
            caPath,
            error: `bridge tls: failed to load cert (${String(err)})`,
        };
    }
}
