import fs from "node:fs/promises";
import path from "node:path";
import { resolveSessionTranscriptsDirForAgent } from "../config/sessions/paths.js";
import { createSubsystemLogger } from "../logging/subsystem.js";
import { hashText } from "./internal.js";
const log = createSubsystemLogger("memory");
export async function listSessionFilesForAgent(agentId) {
    const dir = resolveSessionTranscriptsDirForAgent(agentId);
    try {
        const entries = await fs.readdir(dir, { withFileTypes: true });
        return entries
            .filter((entry) => entry.isFile())
            .map((entry) => entry.name)
            .filter((name) => name.endsWith(".jsonl"))
            .map((name) => path.join(dir, name));
    }
    catch {
        return [];
    }
}
export function sessionPathForFile(absPath) {
    return path.join("sessions", path.basename(absPath)).replace(/\\/g, "/");
}
function normalizeSessionText(value) {
    return value
        .replace(/\s*\n+\s*/g, " ")
        .replace(/\s+/g, " ")
        .trim();
}
export function extractSessionText(content) {
    if (typeof content === "string") {
        const normalized = normalizeSessionText(content);
        return normalized ? normalized : null;
    }
    if (!Array.isArray(content))
        return null;
    const parts = [];
    for (const block of content) {
        if (!block || typeof block !== "object")
            continue;
        const record = block;
        if (record.type !== "text" || typeof record.text !== "string")
            continue;
        const normalized = normalizeSessionText(record.text);
        if (normalized)
            parts.push(normalized);
    }
    if (parts.length === 0)
        return null;
    return parts.join(" ");
}
export async function buildSessionEntry(absPath) {
    try {
        const stat = await fs.stat(absPath);
        const raw = await fs.readFile(absPath, "utf-8");
        const lines = raw.split("\n");
        const collected = [];
        for (const line of lines) {
            if (!line.trim())
                continue;
            let record;
            try {
                record = JSON.parse(line);
            }
            catch {
                continue;
            }
            if (!record ||
                typeof record !== "object" ||
                record.type !== "message") {
                continue;
            }
            const message = record.message;
            if (!message || typeof message.role !== "string")
                continue;
            if (message.role !== "user" && message.role !== "assistant")
                continue;
            const text = extractSessionText(message.content);
            if (!text)
                continue;
            const label = message.role === "user" ? "User" : "Assistant";
            collected.push(`${label}: ${text}`);
        }
        const content = collected.join("\n");
        return {
            path: sessionPathForFile(absPath),
            absPath,
            mtimeMs: stat.mtimeMs,
            size: stat.size,
            hash: hashText(content),
            content,
        };
    }
    catch (err) {
        log.debug(`Failed reading session file ${absPath}: ${String(err)}`);
        return null;
    }
}
