import { DEFAULT_ACCOUNT_ID, normalizeAccountId, } from "../../../routing/session-key.js";
import { listSlackAccountIds, resolveDefaultSlackAccountId, resolveSlackAccount, } from "../../../slack/accounts.js";
import { formatDocsLink } from "../../../terminal/links.js";
import { addWildcardAllowFrom, promptAccountId } from "./helpers.js";
const provider = "slack";
function setSlackDmPolicy(cfg, dmPolicy) {
    const allowFrom = dmPolicy === "open"
        ? addWildcardAllowFrom(cfg.slack?.dm?.allowFrom)
        : undefined;
    return {
        ...cfg,
        slack: {
            ...cfg.slack,
            dm: {
                ...cfg.slack?.dm,
                enabled: cfg.slack?.dm?.enabled ?? true,
                policy: dmPolicy,
                ...(allowFrom ? { allowFrom } : {}),
            },
        },
    };
}
function buildSlackManifest(botName) {
    const safeName = botName.trim() || "Clawdbot";
    const manifest = {
        display_information: {
            name: safeName,
            description: `${safeName} connector for Clawdbot`,
        },
        features: {
            bot_user: {
                display_name: safeName,
                always_online: false,
            },
            app_home: {
                messages_tab_enabled: true,
                messages_tab_read_only_enabled: false,
            },
            slash_commands: [
                {
                    command: "/clawd",
                    description: "Send a message to Clawdbot",
                    should_escape: false,
                },
            ],
        },
        oauth_config: {
            scopes: {
                bot: [
                    "chat:write",
                    "channels:history",
                    "channels:read",
                    "groups:history",
                    "im:history",
                    "mpim:history",
                    "users:read",
                    "app_mentions:read",
                    "reactions:read",
                    "reactions:write",
                    "pins:read",
                    "pins:write",
                    "emoji:read",
                    "commands",
                    "files:read",
                    "files:write",
                ],
            },
        },
        settings: {
            socket_mode_enabled: true,
            event_subscriptions: {
                bot_events: [
                    "app_mention",
                    "message.channels",
                    "message.groups",
                    "message.im",
                    "message.mpim",
                    "reaction_added",
                    "reaction_removed",
                    "member_joined_channel",
                    "member_left_channel",
                    "channel_rename",
                    "pin_added",
                    "pin_removed",
                ],
            },
        },
    };
    return JSON.stringify(manifest, null, 2);
}
async function noteSlackTokenHelp(prompter, botName) {
    const manifest = buildSlackManifest(botName);
    await prompter.note([
        "1) Slack API → Create App → From scratch",
        "2) Add Socket Mode + enable it to get the app-level token (xapp-...)",
        "3) OAuth & Permissions → install app to workspace (xoxb- bot token)",
        "4) Enable Event Subscriptions (socket) for message events",
        "5) App Home → enable the Messages tab for DMs",
        "Tip: set SLACK_BOT_TOKEN + SLACK_APP_TOKEN in your env.",
        `Docs: ${formatDocsLink("/slack", "slack")}`,
        "",
        "Manifest (JSON):",
        manifest,
    ].join("\n"), "Slack socket mode tokens");
}
const dmPolicy = {
    label: "Slack",
    provider,
    policyKey: "slack.dm.policy",
    allowFromKey: "slack.dm.allowFrom",
    getCurrent: (cfg) => cfg.slack?.dm?.policy ?? "pairing",
    setPolicy: (cfg, policy) => setSlackDmPolicy(cfg, policy),
};
export const slackOnboardingAdapter = {
    provider,
    getStatus: async ({ cfg }) => {
        const configured = listSlackAccountIds(cfg).some((accountId) => {
            const account = resolveSlackAccount({ cfg, accountId });
            return Boolean(account.botToken && account.appToken);
        });
        return {
            provider,
            configured,
            statusLines: [`Slack: ${configured ? "configured" : "needs tokens"}`],
            selectionHint: configured ? "configured" : "needs tokens",
            quickstartScore: configured ? 2 : 1,
        };
    },
    configure: async ({ cfg, prompter, accountOverrides, shouldPromptAccountIds, }) => {
        const slackOverride = accountOverrides.slack?.trim();
        const defaultSlackAccountId = resolveDefaultSlackAccountId(cfg);
        let slackAccountId = slackOverride
            ? normalizeAccountId(slackOverride)
            : defaultSlackAccountId;
        if (shouldPromptAccountIds && !slackOverride) {
            slackAccountId = await promptAccountId({
                cfg,
                prompter,
                label: "Slack",
                currentId: slackAccountId,
                listAccountIds: listSlackAccountIds,
                defaultAccountId: defaultSlackAccountId,
            });
        }
        let next = cfg;
        const resolvedAccount = resolveSlackAccount({
            cfg: next,
            accountId: slackAccountId,
        });
        const accountConfigured = Boolean(resolvedAccount.botToken && resolvedAccount.appToken);
        const allowEnv = slackAccountId === DEFAULT_ACCOUNT_ID;
        const canUseEnv = allowEnv &&
            Boolean(process.env.SLACK_BOT_TOKEN?.trim()) &&
            Boolean(process.env.SLACK_APP_TOKEN?.trim());
        const hasConfigTokens = Boolean(resolvedAccount.config.botToken && resolvedAccount.config.appToken);
        let botToken = null;
        let appToken = null;
        const slackBotName = String(await prompter.text({
            message: "Slack bot display name (used for manifest)",
            initialValue: "Clawdbot",
        })).trim();
        if (!accountConfigured) {
            await noteSlackTokenHelp(prompter, slackBotName);
        }
        if (canUseEnv &&
            (!resolvedAccount.config.botToken || !resolvedAccount.config.appToken)) {
            const keepEnv = await prompter.confirm({
                message: "SLACK_BOT_TOKEN + SLACK_APP_TOKEN detected. Use env vars?",
                initialValue: true,
            });
            if (keepEnv) {
                next = {
                    ...next,
                    slack: {
                        ...next.slack,
                        enabled: true,
                    },
                };
            }
            else {
                botToken = String(await prompter.text({
                    message: "Enter Slack bot token (xoxb-...)",
                    validate: (value) => (value?.trim() ? undefined : "Required"),
                })).trim();
                appToken = String(await prompter.text({
                    message: "Enter Slack app token (xapp-...)",
                    validate: (value) => (value?.trim() ? undefined : "Required"),
                })).trim();
            }
        }
        else if (hasConfigTokens) {
            const keep = await prompter.confirm({
                message: "Slack tokens already configured. Keep them?",
                initialValue: true,
            });
            if (!keep) {
                botToken = String(await prompter.text({
                    message: "Enter Slack bot token (xoxb-...)",
                    validate: (value) => (value?.trim() ? undefined : "Required"),
                })).trim();
                appToken = String(await prompter.text({
                    message: "Enter Slack app token (xapp-...)",
                    validate: (value) => (value?.trim() ? undefined : "Required"),
                })).trim();
            }
        }
        else {
            botToken = String(await prompter.text({
                message: "Enter Slack bot token (xoxb-...)",
                validate: (value) => (value?.trim() ? undefined : "Required"),
            })).trim();
            appToken = String(await prompter.text({
                message: "Enter Slack app token (xapp-...)",
                validate: (value) => (value?.trim() ? undefined : "Required"),
            })).trim();
        }
        if (botToken && appToken) {
            if (slackAccountId === DEFAULT_ACCOUNT_ID) {
                next = {
                    ...next,
                    slack: {
                        ...next.slack,
                        enabled: true,
                        botToken,
                        appToken,
                    },
                };
            }
            else {
                next = {
                    ...next,
                    slack: {
                        ...next.slack,
                        enabled: true,
                        accounts: {
                            ...next.slack?.accounts,
                            [slackAccountId]: {
                                ...next.slack?.accounts?.[slackAccountId],
                                enabled: next.slack?.accounts?.[slackAccountId]?.enabled ?? true,
                                botToken,
                                appToken,
                            },
                        },
                    },
                };
            }
        }
        return { cfg: next, accountId: slackAccountId };
    },
    dmPolicy,
    disable: (cfg) => ({
        ...cfg,
        slack: { ...cfg.slack, enabled: false },
    }),
};
