import { formatAge } from "../infra/channel-summary.js";
import { formatTokenCount } from "../utils/usage-format.js";
import { formatContextUsageLine } from "./tui-formatters.js";
export function formatStatusSummary(summary) {
    const lines = [];
    lines.push("Gateway status");
    if (!summary.linkChannel) {
        lines.push("Link channel: unknown");
    }
    else {
        const linkLabel = summary.linkChannel.label ?? "Link channel";
        const linked = summary.linkChannel.linked === true;
        const authAge = linked && typeof summary.linkChannel.authAgeMs === "number"
            ? ` (last refreshed ${formatAge(summary.linkChannel.authAgeMs)})`
            : "";
        lines.push(`${linkLabel}: ${linked ? "linked" : "not linked"}${authAge}`);
    }
    const providerSummary = Array.isArray(summary.providerSummary) ? summary.providerSummary : [];
    if (providerSummary.length > 0) {
        lines.push("");
        lines.push("System:");
        for (const line of providerSummary) {
            lines.push(`  ${line}`);
        }
    }
    const heartbeatAgents = summary.heartbeat?.agents ?? [];
    if (heartbeatAgents.length > 0) {
        const heartbeatParts = heartbeatAgents.map((agent) => {
            const agentId = agent.agentId ?? "unknown";
            if (!agent.enabled || !agent.everyMs)
                return `disabled (${agentId})`;
            return `${agent.every ?? "unknown"} (${agentId})`;
        });
        lines.push("");
        lines.push(`Heartbeat: ${heartbeatParts.join(", ")}`);
    }
    const sessionPaths = summary.sessions?.paths ?? [];
    if (sessionPaths.length === 1) {
        lines.push(`Session store: ${sessionPaths[0]}`);
    }
    else if (sessionPaths.length > 1) {
        lines.push(`Session stores: ${sessionPaths.length}`);
    }
    const defaults = summary.sessions?.defaults;
    const defaultModel = defaults?.model ?? "unknown";
    const defaultCtx = typeof defaults?.contextTokens === "number"
        ? ` (${formatTokenCount(defaults.contextTokens)} ctx)`
        : "";
    lines.push(`Default model: ${defaultModel}${defaultCtx}`);
    const sessionCount = summary.sessions?.count ?? 0;
    lines.push(`Active sessions: ${sessionCount}`);
    const recent = Array.isArray(summary.sessions?.recent) ? summary.sessions?.recent : [];
    if (recent.length > 0) {
        lines.push("Recent sessions:");
        for (const entry of recent) {
            const ageLabel = typeof entry.age === "number" ? formatAge(entry.age) : "no activity";
            const model = entry.model ?? "unknown";
            const usage = formatContextUsageLine({
                total: entry.totalTokens ?? null,
                context: entry.contextTokens ?? null,
                remaining: entry.remainingTokens ?? null,
                percent: entry.percentUsed ?? null,
            });
            const flags = entry.flags?.length ? ` | flags: ${entry.flags.join(", ")}` : "";
            lines.push(`- ${entry.key}${entry.kind ? ` [${entry.kind}]` : ""} | ${ageLabel} | model ${model} | ${usage}${flags}`);
        }
    }
    const queued = Array.isArray(summary.queuedSystemEvents) ? summary.queuedSystemEvents : [];
    if (queued.length > 0) {
        const preview = queued.slice(0, 3).join(" | ");
        lines.push(`Queued system events (${queued.length}): ${preview}`);
    }
    return lines;
}
