import { CastableBase, ReadonlyArray, ReadonlyPath, type JsonArray, type JsonObject, type array, type merge, type propwiseXor, type show } from "@ark/util";
import type { Prerequisite, errorContext } from "../kinds.ts";
import type { NodeKind } from "./implement.ts";
import type { StandardSchemaV1 } from "./standardSchema.ts";
import type { Traversal } from "./traversal.ts";
import { arkKind } from "./utils.ts";
export type ArkErrorResult = ArkError | ArkErrors;
export declare class ArkError<code extends ArkErrorCode = ArkErrorCode> extends CastableBase<ArkErrorContextInput<code>> {
    readonly [arkKind] = "error";
    path: ReadonlyPath;
    data: Prerequisite<code>;
    private nodeConfig;
    protected input: ArkErrorContextInput<code>;
    protected ctx: Traversal;
    constructor(input: ArkErrorContextInput<code>, ctx: Traversal);
    transform(f: (input: ArkErrorContextInput<code>) => ArkErrorContextInput): ArkError;
    hasCode<code extends ArkErrorCode>(code: code): this is ArkError<code>;
    get propString(): string;
    get expected(): string;
    get actual(): string;
    get problem(): string;
    get message(): string;
    get flat(): ArkError[];
    toJSON(): JsonObject;
    toString(): string;
    throw(): never;
}
export declare namespace ArkErrors {
    type Handler<returns = unknown> = (errors: ArkErrors) => returns;
}
/**
 * A ReadonlyArray of `ArkError`s returned by a Type on invalid input.
 *
 * Subsequent errors added at an existing path are merged into an
 * ArkError intersection.
 */
export declare class ArkErrors extends ReadonlyArray<ArkError> implements StandardSchemaV1.FailureResult {
    readonly [arkKind] = "errors";
    protected ctx: Traversal;
    constructor(ctx: Traversal);
    /**
     * Errors by a pathString representing their location.
     */
    byPath: Record<string, ArkError>;
    /**
     * {@link byPath} flattened so that each value is an array of ArkError instances at that path.
     *
     * ✅ Since "intersection" errors will be flattened to their constituent `.errors`,
     * they will never be directly present in this representation.
     */
    get flatByPath(): Record<string, ArkError[]>;
    /**
     * {@link byPath} flattened so that each value is an array of problem strings at that path.
     */
    get flatProblemsByPath(): Record<string, string[]>;
    /**
     * All pathStrings at which errors are present mapped to the errors occuring
     * at that path or any nested path within it.
     */
    byAncestorPath: Record<string, ArkError[]>;
    count: number;
    private mutable;
    /**
     * Throw a TraversalError based on these errors.
     */
    throw(): never;
    /**
     * Converts ArkErrors to TraversalError, a subclass of `Error` suitable for throwing with nice
     * formatting.
     */
    toTraversalError(): TraversalError;
    /**
     * Append an ArkError to this array, ignoring duplicates.
     */
    add(error: ArkError): void;
    transform(f: (e: ArkError) => ArkError): ArkErrors;
    /**
     * Add all errors from an ArkErrors instance, ignoring duplicates and
     * prefixing their paths with that of the current Traversal.
     */
    merge(errors: ArkErrors): void;
    /**
     * @internal
     */
    affectsPath(path: ReadonlyPath): boolean;
    /**
     * A human-readable summary of all errors.
     */
    get summary(): string;
    /**
     * Alias of this ArkErrors instance for StandardSchema compatibility.
     */
    get issues(): this;
    toJSON(): JsonArray;
    toString(): string;
    private addAncestorPaths;
}
export declare class TraversalError extends Error {
    readonly name = "TraversalError";
    arkErrors: ArkErrors;
    constructor(errors: ArkErrors);
}
export interface DerivableErrorContext<code extends ArkErrorCode = ArkErrorCode> {
    expected: string;
    actual: string;
    problem: string;
    message: string;
    data: Prerequisite<code>;
    path: array<PropertyKey>;
    propString: string;
}
export type DerivableErrorContextInput<code extends ArkErrorCode = ArkErrorCode> = Partial<DerivableErrorContext<code>> & propwiseXor<{
    path?: array<PropertyKey>;
}, {
    relativePath?: array<PropertyKey>;
    prefixPath?: array<PropertyKey>;
}>;
export type ArkErrorCode = {
    [kind in NodeKind]: errorContext<kind> extends null ? never : kind;
}[NodeKind];
type ArkErrorContextInputsByCode = {
    [code in ArkErrorCode]: errorContext<code> & DerivableErrorContextInput<code>;
};
export type ArkErrorContextInput<code extends ArkErrorCode = ArkErrorCode> = merge<ArkErrorContextInputsByCode[code], {
    meta?: ArkEnv.meta;
}>;
export type NodeErrorContextInput<code extends ArkErrorCode = ArkErrorCode> = ArkErrorContextInputsByCode[code] & {
    meta: ArkEnv.meta;
};
export type MessageContext<code extends ArkErrorCode = ArkErrorCode> = Omit<ArkError<code>, "message">;
export type ProblemContext<code extends ArkErrorCode = ArkErrorCode> = Omit<MessageContext<code>, "problem">;
export type CustomErrorInput = show<{
    code?: undefined;
} & DerivableErrorContextInput>;
export type ArkErrorInput = string | ArkErrorContextInput | CustomErrorInput;
export type ProblemConfig<code extends ArkErrorCode = ArkErrorCode> = string | ProblemWriter<code>;
export type ProblemWriter<code extends ArkErrorCode = ArkErrorCode> = (context: ProblemContext<code>) => string;
export type MessageConfig<code extends ArkErrorCode = ArkErrorCode> = string | MessageWriter<code>;
export type MessageWriter<code extends ArkErrorCode = ArkErrorCode> = (context: MessageContext<code>) => string;
export type getAssociatedDataForError<code extends ArkErrorCode> = code extends NodeKind ? Prerequisite<code> : unknown;
export type ExpectedConfig<code extends ArkErrorCode = ArkErrorCode> = string | ExpectedWriter<code>;
export type ExpectedWriter<code extends ArkErrorCode = ArkErrorCode> = (source: errorContext<code>) => string;
export type ActualConfig<code extends ArkErrorCode = ArkErrorCode> = string | ActualWriter<code>;
export type ActualWriter<code extends ArkErrorCode = ArkErrorCode> = (data: getAssociatedDataForError<code>) => string;
export {};
