// NEVER convert to top-level imports - breaks browser/Vite builds (web-ui)
let _existsSync = null;
let _homedir = null;
let _join = null;
// Eagerly load in Node.js environment only
if (typeof process !== "undefined" && process.versions?.node) {
    import("node:fs").then((m) => {
        _existsSync = m.existsSync;
    });
    import("node:os").then((m) => {
        _homedir = m.homedir;
    });
    import("node:path").then((m) => {
        _join = m.join;
    });
}
import { supportsXhigh } from "./models.js";
import { streamBedrock } from "./providers/amazon-bedrock.js";
import { streamAnthropic } from "./providers/anthropic.js";
import { streamGoogle } from "./providers/google.js";
import { streamGoogleGeminiCli, } from "./providers/google-gemini-cli.js";
import { streamGoogleVertex } from "./providers/google-vertex.js";
import { streamOpenAICodexResponses } from "./providers/openai-codex-responses.js";
import { streamOpenAICompletions } from "./providers/openai-completions.js";
import { streamOpenAIResponses } from "./providers/openai-responses.js";
let cachedVertexAdcCredentialsExists = null;
function hasVertexAdcCredentials() {
    if (cachedVertexAdcCredentialsExists === null) {
        // In browser or if node modules not loaded yet, return false
        if (!_existsSync || !_homedir || !_join) {
            cachedVertexAdcCredentialsExists = false;
            return false;
        }
        // Check GOOGLE_APPLICATION_CREDENTIALS env var first (standard way)
        const gacPath = process.env.GOOGLE_APPLICATION_CREDENTIALS;
        if (gacPath) {
            cachedVertexAdcCredentialsExists = _existsSync(gacPath);
        }
        else {
            // Fall back to default ADC path (lazy evaluation)
            cachedVertexAdcCredentialsExists = _existsSync(_join(_homedir(), ".config", "gcloud", "application_default_credentials.json"));
        }
    }
    return cachedVertexAdcCredentialsExists;
}
export function getEnvApiKey(provider) {
    // Fall back to environment variables
    if (provider === "github-copilot") {
        return process.env.COPILOT_GITHUB_TOKEN || process.env.GH_TOKEN || process.env.GITHUB_TOKEN;
    }
    // ANTHROPIC_OAUTH_TOKEN takes precedence over ANTHROPIC_API_KEY
    if (provider === "anthropic") {
        return process.env.ANTHROPIC_OAUTH_TOKEN || process.env.ANTHROPIC_API_KEY;
    }
    // Vertex AI uses Application Default Credentials, not API keys.
    // Auth is configured via `gcloud auth application-default login`.
    if (provider === "google-vertex") {
        const hasCredentials = hasVertexAdcCredentials();
        const hasProject = !!(process.env.GOOGLE_CLOUD_PROJECT || process.env.GCLOUD_PROJECT);
        const hasLocation = !!process.env.GOOGLE_CLOUD_LOCATION;
        if (hasCredentials && hasProject && hasLocation) {
            return "<authenticated>";
        }
    }
    if (provider === "amazon-bedrock") {
        // Amazon Bedrock supports multiple credential sources:
        // 1. AWS_PROFILE - named profile from ~/.aws/credentials
        // 2. AWS_ACCESS_KEY_ID + AWS_SECRET_ACCESS_KEY - standard IAM keys
        // 3. AWS_BEARER_TOKEN_BEDROCK - Bedrock API keys (bearer token)
        // 4. AWS_CONTAINER_CREDENTIALS_RELATIVE_URI - ECS task roles
        // 5. AWS_CONTAINER_CREDENTIALS_FULL_URI - ECS task roles (full URI)
        // 6. AWS_WEB_IDENTITY_TOKEN_FILE - IRSA (IAM Roles for Service Accounts)
        if (process.env.AWS_PROFILE ||
            (process.env.AWS_ACCESS_KEY_ID && process.env.AWS_SECRET_ACCESS_KEY) ||
            process.env.AWS_BEARER_TOKEN_BEDROCK ||
            process.env.AWS_CONTAINER_CREDENTIALS_RELATIVE_URI ||
            process.env.AWS_CONTAINER_CREDENTIALS_FULL_URI ||
            process.env.AWS_WEB_IDENTITY_TOKEN_FILE) {
            return "<authenticated>";
        }
    }
    const envMap = {
        openai: "OPENAI_API_KEY",
        google: "GEMINI_API_KEY",
        groq: "GROQ_API_KEY",
        cerebras: "CEREBRAS_API_KEY",
        xai: "XAI_API_KEY",
        openrouter: "OPENROUTER_API_KEY",
        "vercel-ai-gateway": "AI_GATEWAY_API_KEY",
        zai: "ZAI_API_KEY",
        mistral: "MISTRAL_API_KEY",
        minimax: "MINIMAX_API_KEY",
        "minimax-cn": "MINIMAX_CN_API_KEY",
        opencode: "OPENCODE_API_KEY",
    };
    const envVar = envMap[provider];
    return envVar ? process.env[envVar] : undefined;
}
export function stream(model, context, options) {
    // Vertex AI uses Application Default Credentials, not API keys
    if (model.api === "google-vertex") {
        return streamGoogleVertex(model, context, options);
    }
    else if (model.api === "bedrock-converse-stream") {
        // Bedrock doesn't have any API keys instead it sources credentials from standard AWS env variables or from given AWS profile.
        return streamBedrock(model, context, (options || {}));
    }
    const apiKey = options?.apiKey || getEnvApiKey(model.provider);
    if (!apiKey) {
        throw new Error(`No API key for provider: ${model.provider}`);
    }
    const providerOptions = { ...options, apiKey };
    const api = model.api;
    switch (api) {
        case "anthropic-messages":
            return streamAnthropic(model, context, providerOptions);
        case "openai-completions":
            return streamOpenAICompletions(model, context, providerOptions);
        case "openai-responses":
            return streamOpenAIResponses(model, context, providerOptions);
        case "openai-codex-responses":
            return streamOpenAICodexResponses(model, context, providerOptions);
        case "google-generative-ai":
            return streamGoogle(model, context, providerOptions);
        case "google-gemini-cli":
            return streamGoogleGeminiCli(model, context, providerOptions);
        default: {
            // This should never be reached if all Api cases are handled
            const _exhaustive = api;
            throw new Error(`Unhandled API: ${_exhaustive}`);
        }
    }
}
export async function complete(model, context, options) {
    const s = stream(model, context, options);
    return s.result();
}
export function streamSimple(model, context, options) {
    // Vertex AI uses Application Default Credentials, not API keys
    if (model.api === "google-vertex") {
        const providerOptions = mapOptionsForApi(model, options, undefined);
        return stream(model, context, providerOptions);
    }
    else if (model.api === "bedrock-converse-stream") {
        // Bedrock doesn't have any API keys instead it sources credentials from standard AWS env variables or from given AWS profile.
        const providerOptions = mapOptionsForApi(model, options, undefined);
        return stream(model, context, providerOptions);
    }
    const apiKey = options?.apiKey || getEnvApiKey(model.provider);
    if (!apiKey) {
        throw new Error(`No API key for provider: ${model.provider}`);
    }
    const providerOptions = mapOptionsForApi(model, options, apiKey);
    return stream(model, context, providerOptions);
}
export async function completeSimple(model, context, options) {
    const s = streamSimple(model, context, options);
    return s.result();
}
function mapOptionsForApi(model, options, apiKey) {
    const base = {
        temperature: options?.temperature,
        maxTokens: options?.maxTokens || Math.min(model.maxTokens, 32000),
        signal: options?.signal,
        apiKey: apiKey || options?.apiKey,
        sessionId: options?.sessionId,
        headers: options?.headers,
        onPayload: options?.onPayload,
    };
    // Helper to clamp xhigh to high for providers that don't support it
    const clampReasoning = (effort) => (effort === "xhigh" ? "high" : effort);
    /**
     * Adjust maxTokens to account for thinking budget.
     * APIs like Anthropic and Bedrock require max_tokens > thinking.budget_tokens.
     * Returns { adjustedMaxTokens, adjustedThinkingBudget }
     */
    const adjustMaxTokensForThinking = (baseMaxTokens, modelMaxTokens, reasoningLevel, customBudgets) => {
        const defaultBudgets = {
            minimal: 1024,
            low: 2048,
            medium: 8192,
            high: 16384,
        };
        const budgets = { ...defaultBudgets, ...customBudgets };
        const minOutputTokens = 1024;
        const level = clampReasoning(reasoningLevel);
        let thinkingBudget = budgets[level];
        // Caller's maxTokens is the desired output; add thinking budget on top, capped at model limit
        const maxTokens = Math.min(baseMaxTokens + thinkingBudget, modelMaxTokens);
        // If not enough room for thinking + output, reduce thinking budget
        if (maxTokens <= thinkingBudget) {
            thinkingBudget = Math.max(0, maxTokens - minOutputTokens);
        }
        return { maxTokens, thinkingBudget };
    };
    switch (model.api) {
        case "anthropic-messages": {
            // Explicitly disable thinking when reasoning is not specified
            if (!options?.reasoning) {
                return { ...base, thinkingEnabled: false };
            }
            // Claude requires max_tokens > thinking.budget_tokens
            // So we need to ensure maxTokens accounts for both thinking and output
            const adjusted = adjustMaxTokensForThinking(base.maxTokens || 0, model.maxTokens, options.reasoning, options?.thinkingBudgets);
            return {
                ...base,
                maxTokens: adjusted.maxTokens,
                thinkingEnabled: true,
                thinkingBudgetTokens: adjusted.thinkingBudget,
            };
        }
        case "bedrock-converse-stream": {
            // Explicitly disable thinking when reasoning is not specified
            if (!options?.reasoning) {
                return { ...base, reasoning: undefined };
            }
            // Claude requires max_tokens > thinking.budget_tokens (same as Anthropic direct API)
            // So we need to ensure maxTokens accounts for both thinking and output
            if (model.id.includes("anthropic.claude") || model.id.includes("anthropic/claude")) {
                const adjusted = adjustMaxTokensForThinking(base.maxTokens || 0, model.maxTokens, options.reasoning, options?.thinkingBudgets);
                return {
                    ...base,
                    maxTokens: adjusted.maxTokens,
                    reasoning: options.reasoning,
                    thinkingBudgets: {
                        ...(options?.thinkingBudgets || {}),
                        [clampReasoning(options.reasoning)]: adjusted.thinkingBudget,
                    },
                };
            }
            // Non-Claude models - pass through
            return {
                ...base,
                reasoning: options?.reasoning,
                thinkingBudgets: options?.thinkingBudgets,
            };
        }
        case "openai-completions":
            return {
                ...base,
                reasoningEffort: supportsXhigh(model) ? options?.reasoning : clampReasoning(options?.reasoning),
            };
        case "openai-responses":
            return {
                ...base,
                reasoningEffort: supportsXhigh(model) ? options?.reasoning : clampReasoning(options?.reasoning),
            };
        case "openai-codex-responses":
            return {
                ...base,
                reasoningEffort: supportsXhigh(model) ? options?.reasoning : clampReasoning(options?.reasoning),
            };
        case "google-generative-ai": {
            // Explicitly disable thinking when reasoning is not specified
            // This is needed because Gemini has "dynamic thinking" enabled by default
            if (!options?.reasoning) {
                return { ...base, thinking: { enabled: false } };
            }
            const googleModel = model;
            const effort = clampReasoning(options.reasoning);
            // Gemini 3 models use thinkingLevel exclusively instead of thinkingBudget.
            // https://ai.google.dev/gemini-api/docs/thinking#set-budget
            if (isGemini3ProModel(googleModel) || isGemini3FlashModel(googleModel)) {
                return {
                    ...base,
                    thinking: {
                        enabled: true,
                        level: getGemini3ThinkingLevel(effort, googleModel),
                    },
                };
            }
            return {
                ...base,
                thinking: {
                    enabled: true,
                    budgetTokens: getGoogleBudget(googleModel, effort, options?.thinkingBudgets),
                },
            };
        }
        case "google-gemini-cli": {
            if (!options?.reasoning) {
                return { ...base, thinking: { enabled: false } };
            }
            const effort = clampReasoning(options.reasoning);
            // Gemini 3 models use thinkingLevel instead of thinkingBudget
            if (model.id.includes("3-pro") || model.id.includes("3-flash")) {
                return {
                    ...base,
                    thinking: {
                        enabled: true,
                        level: getGeminiCliThinkingLevel(effort, model.id),
                    },
                };
            }
            // Models using thinkingBudget (Gemini 2.x, Claude via Antigravity)
            // Claude requires max_tokens > thinking.budget_tokens
            // So we need to ensure maxTokens accounts for both thinking and output
            const defaultBudgets = {
                minimal: 1024,
                low: 2048,
                medium: 8192,
                high: 16384,
            };
            const budgets = { ...defaultBudgets, ...options?.thinkingBudgets };
            const minOutputTokens = 1024;
            let thinkingBudget = budgets[effort];
            // Caller's maxTokens is the desired output; add thinking budget on top, capped at model limit
            const maxTokens = Math.min((base.maxTokens || 0) + thinkingBudget, model.maxTokens);
            // If not enough room for thinking + output, reduce thinking budget
            if (maxTokens <= thinkingBudget) {
                thinkingBudget = Math.max(0, maxTokens - minOutputTokens);
            }
            return {
                ...base,
                maxTokens,
                thinking: {
                    enabled: true,
                    budgetTokens: thinkingBudget,
                },
            };
        }
        case "google-vertex": {
            // Explicitly disable thinking when reasoning is not specified
            if (!options?.reasoning) {
                return { ...base, thinking: { enabled: false } };
            }
            const vertexModel = model;
            const effort = clampReasoning(options.reasoning);
            const geminiModel = vertexModel;
            if (isGemini3ProModel(geminiModel) || isGemini3FlashModel(geminiModel)) {
                return {
                    ...base,
                    thinking: {
                        enabled: true,
                        level: getGemini3ThinkingLevel(effort, geminiModel),
                    },
                };
            }
            return {
                ...base,
                thinking: {
                    enabled: true,
                    budgetTokens: getGoogleBudget(geminiModel, effort, options?.thinkingBudgets),
                },
            };
        }
        default: {
            // Exhaustiveness check
            const _exhaustive = model.api;
            throw new Error(`Unhandled API in mapOptionsForApi: ${_exhaustive}`);
        }
    }
}
function isGemini3ProModel(model) {
    // Covers gemini-3-pro, gemini-3-pro-preview, and possible other prefixed ids in the future
    return model.id.includes("3-pro");
}
function isGemini3FlashModel(model) {
    // Covers gemini-3-flash, gemini-3-flash-preview, and possible other prefixed ids in the future
    return model.id.includes("3-flash");
}
function getGemini3ThinkingLevel(effort, model) {
    if (isGemini3ProModel(model)) {
        // Gemini 3 Pro only supports LOW/HIGH (for now)
        switch (effort) {
            case "minimal":
            case "low":
                return "LOW";
            case "medium":
            case "high":
                return "HIGH";
        }
    }
    // Gemini 3 Flash supports all four levels
    switch (effort) {
        case "minimal":
            return "MINIMAL";
        case "low":
            return "LOW";
        case "medium":
            return "MEDIUM";
        case "high":
            return "HIGH";
    }
}
function getGeminiCliThinkingLevel(effort, modelId) {
    if (modelId.includes("3-pro")) {
        // Gemini 3 Pro only supports LOW/HIGH (for now)
        switch (effort) {
            case "minimal":
            case "low":
                return "LOW";
            case "medium":
            case "high":
                return "HIGH";
        }
    }
    // Gemini 3 Flash supports all four levels
    switch (effort) {
        case "minimal":
            return "MINIMAL";
        case "low":
            return "LOW";
        case "medium":
            return "MEDIUM";
        case "high":
            return "HIGH";
    }
}
function getGoogleBudget(model, effort, customBudgets) {
    // Custom budgets take precedence if provided for this level
    if (customBudgets?.[effort] !== undefined) {
        return customBudgets[effort];
    }
    // See https://ai.google.dev/gemini-api/docs/thinking#set-budget
    if (model.id.includes("2.5-pro")) {
        const budgets = {
            minimal: 128,
            low: 2048,
            medium: 8192,
            high: 32768,
        };
        return budgets[effort];
    }
    if (model.id.includes("2.5-flash")) {
        // Covers 2.5-flash-lite as well
        const budgets = {
            minimal: 128,
            low: 2048,
            medium: 8192,
            high: 24576,
        };
        return budgets[effort];
    }
    // Unknown model - use dynamic
    return -1;
}
//# sourceMappingURL=stream.js.map