/**
 * OAuth credential management for AI providers.
 *
 * This module handles login, token refresh, and credential storage
 * for OAuth-based providers:
 * - Anthropic (Claude Pro/Max)
 * - GitHub Copilot
 * - Google Cloud Code Assist (Gemini CLI)
 * - Antigravity (Gemini 3, Claude, GPT-OSS via Google Cloud)
 */
export { loginAnthropic, refreshAnthropicToken } from "./anthropic.js";
export { getGitHubCopilotBaseUrl, loginGitHubCopilot, normalizeDomain, refreshGitHubCopilotToken, } from "./github-copilot.js";
export { loginAntigravity, refreshAntigravityToken, } from "./google-antigravity.js";
export { loginGeminiCli, refreshGoogleCloudToken, } from "./google-gemini-cli.js";
export { loginOpenAICodex, refreshOpenAICodexToken, } from "./openai-codex.js";
export * from "./types.js";
import type { OAuthCredentials, OAuthProvider, OAuthProviderInfo } from "./types.js";
/**
 * Refresh token for any OAuth provider.
 * Saves the new credentials and returns the new access token.
 */
export declare function refreshOAuthToken(provider: OAuthProvider, credentials: OAuthCredentials): Promise<OAuthCredentials>;
/**
 * Get API key for a provider from OAuth credentials.
 * Automatically refreshes expired tokens.
 *
 * For google-gemini-cli and antigravity, returns JSON-encoded { token, projectId }
 *
 * @returns API key string, or null if no credentials
 * @throws Error if refresh fails
 */
export declare function getOAuthApiKey(provider: OAuthProvider, credentials: Record<string, OAuthCredentials>): Promise<{
    newCredentials: OAuthCredentials;
    apiKey: string;
} | null>;
/**
 * Get list of OAuth providers
 */
export declare function getOAuthProviders(): OAuthProviderInfo[];
//# sourceMappingURL=index.d.ts.map