/**
 * Model registry - manages built-in and custom models, provides API key resolution.
 */
import { type Api, type Model } from "@mariozechner/pi-ai";
import type { AuthStorage } from "./auth-storage.js";
/** Clear the API key command cache. Exported for testing. */
export declare function clearApiKeyCache(): void;
/**
 * Model registry - loads and manages models, resolves API keys via AuthStorage.
 */
export declare class ModelRegistry {
    readonly authStorage: AuthStorage;
    private modelsJsonPath;
    private models;
    private customProviderApiKeys;
    private loadError;
    constructor(authStorage: AuthStorage, modelsJsonPath?: string | undefined);
    /**
     * Reload models from disk (built-in + custom from models.json).
     */
    refresh(): void;
    /**
     * Get any error from loading models.json (undefined if no error).
     */
    getError(): string | undefined;
    private loadModels;
    /** Load built-in models, skipping replaced providers and applying overrides */
    private loadBuiltInModels;
    private loadCustomModels;
    private validateConfig;
    private parseModels;
    /**
     * Get all models (built-in + custom).
     * If models.json had errors, returns only built-in models.
     */
    getAll(): Model<Api>[];
    /**
     * Get only models that have auth configured.
     * This is a fast check that doesn't refresh OAuth tokens.
     */
    getAvailable(): Model<Api>[];
    /**
     * Find a model by provider and ID.
     */
    find(provider: string, modelId: string): Model<Api> | undefined;
    /**
     * Get API key for a model.
     */
    getApiKey(model: Model<Api>): Promise<string | undefined>;
    /**
     * Get API key for a provider.
     */
    getApiKeyForProvider(provider: string): Promise<string | undefined>;
    /**
     * Check if a model is using OAuth credentials (subscription).
     */
    isUsingOAuth(model: Model<Api>): boolean;
}
//# sourceMappingURL=model-registry.d.ts.map