import { existsSync, readdirSync, readFileSync, statSync } from "fs";
import { join, resolve } from "path";
import { CONFIG_DIR_NAME, getPromptsDir } from "../config.js";
import { parseFrontmatter } from "../utils/frontmatter.js";
/**
 * Parse command arguments respecting quoted strings (bash-style)
 * Returns array of arguments
 */
export function parseCommandArgs(argsString) {
    const args = [];
    let current = "";
    let inQuote = null;
    for (let i = 0; i < argsString.length; i++) {
        const char = argsString[i];
        if (inQuote) {
            if (char === inQuote) {
                inQuote = null;
            }
            else {
                current += char;
            }
        }
        else if (char === '"' || char === "'") {
            inQuote = char;
        }
        else if (char === " " || char === "\t") {
            if (current) {
                args.push(current);
                current = "";
            }
        }
        else {
            current += char;
        }
    }
    if (current) {
        args.push(current);
    }
    return args;
}
/**
 * Substitute argument placeholders in template content
 * Supports:
 * - $1, $2, ... for positional args
 * - $@ and $ARGUMENTS for all args
 * - ${@:N} for args from Nth onwards (bash-style slicing)
 * - ${@:N:L} for L args starting from Nth
 *
 * Note: Replacement happens on the template string only. Argument values
 * containing patterns like $1, $@, or $ARGUMENTS are NOT recursively substituted.
 */
export function substituteArgs(content, args) {
    let result = content;
    // Replace $1, $2, etc. with positional args FIRST (before wildcards)
    // This prevents wildcard replacement values containing $<digit> patterns from being re-substituted
    result = result.replace(/\$(\d+)/g, (_, num) => {
        const index = parseInt(num, 10) - 1;
        return args[index] ?? "";
    });
    // Replace ${@:start} or ${@:start:length} with sliced args (bash-style)
    // Process BEFORE simple $@ to avoid conflicts
    result = result.replace(/\$\{@:(\d+)(?::(\d+))?\}/g, (_, startStr, lengthStr) => {
        let start = parseInt(startStr, 10) - 1; // Convert to 0-indexed (user provides 1-indexed)
        // Treat 0 as 1 (bash convention: args start at 1)
        if (start < 0)
            start = 0;
        if (lengthStr) {
            const length = parseInt(lengthStr, 10);
            return args.slice(start, start + length).join(" ");
        }
        return args.slice(start).join(" ");
    });
    // Pre-compute all args joined (optimization)
    const allArgs = args.join(" ");
    // Replace $ARGUMENTS with all args joined (new syntax, aligns with Claude, Codex, OpenCode)
    result = result.replace(/\$ARGUMENTS/g, allArgs);
    // Replace $@ with all args joined (existing syntax)
    result = result.replace(/\$@/g, allArgs);
    return result;
}
/**
 * Recursively scan a directory for .md files (and symlinks to .md files) and load them as prompt templates
 */
function loadTemplatesFromDir(dir, source, subdir = "") {
    const templates = [];
    if (!existsSync(dir)) {
        return templates;
    }
    try {
        const entries = readdirSync(dir, { withFileTypes: true });
        for (const entry of entries) {
            const fullPath = join(dir, entry.name);
            // For symlinks, check if they point to a directory and follow them
            let isDirectory = entry.isDirectory();
            let isFile = entry.isFile();
            if (entry.isSymbolicLink()) {
                try {
                    const stats = statSync(fullPath);
                    isDirectory = stats.isDirectory();
                    isFile = stats.isFile();
                }
                catch {
                    // Broken symlink, skip it
                    continue;
                }
            }
            if (isDirectory) {
                // Recurse into subdirectory
                const newSubdir = subdir ? `${subdir}:${entry.name}` : entry.name;
                templates.push(...loadTemplatesFromDir(fullPath, source, newSubdir));
            }
            else if (isFile && entry.name.endsWith(".md")) {
                try {
                    const rawContent = readFileSync(fullPath, "utf-8");
                    const { frontmatter, body } = parseFrontmatter(rawContent);
                    const name = entry.name.slice(0, -3); // Remove .md extension
                    // Build source string
                    let sourceStr;
                    if (source === "user") {
                        sourceStr = subdir ? `(user:${subdir})` : "(user)";
                    }
                    else {
                        sourceStr = subdir ? `(project:${subdir})` : "(project)";
                    }
                    // Get description from frontmatter or first non-empty line
                    let description = frontmatter.description || "";
                    if (!description) {
                        const firstLine = body.split("\n").find((line) => line.trim());
                        if (firstLine) {
                            // Truncate if too long
                            description = firstLine.slice(0, 60);
                            if (firstLine.length > 60)
                                description += "...";
                        }
                    }
                    // Append source to description
                    description = description ? `${description} ${sourceStr}` : sourceStr;
                    templates.push({
                        name,
                        description,
                        content: body,
                        source: sourceStr,
                    });
                }
                catch (_error) {
                    // Silently skip files that can't be read
                }
            }
        }
    }
    catch (_error) {
        // Silently skip directories that can't be read
    }
    return templates;
}
/**
 * Load all prompt templates from:
 * 1. Global: agentDir/prompts/
 * 2. Project: cwd/{CONFIG_DIR_NAME}/prompts/
 */
export function loadPromptTemplates(options = {}) {
    const resolvedCwd = options.cwd ?? process.cwd();
    const resolvedAgentDir = options.agentDir ?? getPromptsDir();
    const templates = [];
    // 1. Load global templates from agentDir/prompts/
    // Note: if agentDir is provided, it should be the agent dir, not the prompts dir
    const globalPromptsDir = options.agentDir ? join(options.agentDir, "prompts") : resolvedAgentDir;
    templates.push(...loadTemplatesFromDir(globalPromptsDir, "user"));
    // 2. Load project templates from cwd/{CONFIG_DIR_NAME}/prompts/
    const projectPromptsDir = resolve(resolvedCwd, CONFIG_DIR_NAME, "prompts");
    templates.push(...loadTemplatesFromDir(projectPromptsDir, "project"));
    return templates;
}
/**
 * Expand a prompt template if it matches a template name.
 * Returns the expanded content or the original text if not a template.
 */
export function expandPromptTemplate(text, templates) {
    if (!text.startsWith("/"))
        return text;
    const spaceIndex = text.indexOf(" ");
    const templateName = spaceIndex === -1 ? text.slice(1) : text.slice(1, spaceIndex);
    const argsString = spaceIndex === -1 ? "" : text.slice(spaceIndex + 1);
    const template = templates.find((t) => t.name === templateName);
    if (template) {
        const args = parseCommandArgs(argsString);
        return substituteArgs(template.content, args);
    }
    return text;
}
//# sourceMappingURL=prompt-templates.js.map