/**
 * SDK for programmatic usage of AgentSession.
 *
 * Provides a factory function and discovery helpers that allow full control
 * over agent configuration, or sensible defaults that match CLI behavior.
 *
 * @example
 * ```typescript
 * // Minimal - everything auto-discovered
 * const session = await createAgentSession();
 *
 * // Full control
 * const session = await createAgentSession({
 *   model: myModel,
 *   getApiKey: async () => process.env.MY_KEY,
 *   tools: [readTool, bashTool],
 *   skills: [],
 *   sessionFile: false,
 * });
 * ```
 */
import { type ThinkingLevel } from "@mariozechner/pi-agent-core";
import type { Model } from "@mariozechner/pi-ai";
import { AgentSession } from "./agent-session.js";
import { AuthStorage } from "./auth-storage.js";
import { type EventBus } from "./event-bus.js";
import { type ExtensionFactory, type LoadExtensionsResult, type ToolDefinition } from "./extensions/index.js";
import { ModelRegistry } from "./model-registry.js";
import { type PromptTemplate } from "./prompt-templates.js";
import { SessionManager } from "./session-manager.js";
import { type Settings, SettingsManager, type SkillsSettings } from "./settings-manager.js";
import { type Skill, type SkillWarning } from "./skills.js";
import { allTools, bashTool, codingTools, createBashTool, createCodingTools, createEditTool, createFindTool, createGrepTool, createLsTool, createReadOnlyTools, createReadTool, createWriteTool, editTool, findTool, grepTool, lsTool, readOnlyTools, readTool, type Tool, writeTool } from "./tools/index.js";
export interface CreateAgentSessionOptions {
    /** Working directory for project-local discovery. Default: process.cwd() */
    cwd?: string;
    /** Global config directory. Default: ~/.pi/agent */
    agentDir?: string;
    /** Auth storage for credentials. Default: discoverAuthStorage(agentDir) */
    authStorage?: AuthStorage;
    /** Model registry. Default: discoverModels(authStorage, agentDir) */
    modelRegistry?: ModelRegistry;
    /** Model to use. Default: from settings, else first available */
    model?: Model<any>;
    /** Thinking level. Default: from settings, else 'off' (clamped to model capabilities) */
    thinkingLevel?: ThinkingLevel;
    /** Models available for cycling (Ctrl+P in interactive mode) */
    scopedModels?: Array<{
        model: Model<any>;
        thinkingLevel: ThinkingLevel;
    }>;
    /** System prompt. String replaces default, function receives default and returns final. */
    systemPrompt?: string | ((defaultPrompt: string) => string);
    /** Built-in tools to use. Default: codingTools [read, bash, edit, write] */
    tools?: Tool[];
    /** Custom tools to register (in addition to built-in tools). */
    customTools?: ToolDefinition[];
    /** Inline extensions. When provided (even if empty), skips file discovery. */
    extensions?: ExtensionFactory[];
    /** Additional extension paths to load (merged with discovery). */
    additionalExtensionPaths?: string[];
    /**
     * Pre-loaded extensions result (skips file discovery).
     * @internal Used by CLI when extensions are loaded early to parse custom flags.
     */
    preloadedExtensions?: LoadExtensionsResult;
    /** Shared event bus for tool/extension communication. Default: creates new bus. */
    eventBus?: EventBus;
    /** Skills. Default: discovered from multiple locations */
    skills?: Skill[];
    /** Context files (AGENTS.md content). Default: discovered walking up from cwd */
    contextFiles?: Array<{
        path: string;
        content: string;
    }>;
    /** Prompt templates. Default: discovered from cwd/.pi/prompts/ + agentDir/prompts/ */
    promptTemplates?: PromptTemplate[];
    /** Session manager. Default: SessionManager.create(cwd) */
    sessionManager?: SessionManager;
    /** Settings manager. Default: SettingsManager.create(cwd, agentDir) */
    settingsManager?: SettingsManager;
}
/** Result from createAgentSession */
export interface CreateAgentSessionResult {
    /** The created session */
    session: AgentSession;
    /** Extensions result (for UI context setup in interactive mode) */
    extensionsResult: LoadExtensionsResult;
    /** Warning if session was restored with a different model than saved */
    modelFallbackMessage?: string;
}
export type { ExtensionAPI, ExtensionCommandContext, ExtensionContext, ExtensionFactory, ToolDefinition, } from "./extensions/index.js";
export type { PromptTemplate } from "./prompt-templates.js";
export type { Settings, SkillsSettings } from "./settings-manager.js";
export type { Skill } from "./skills.js";
export type { Tool } from "./tools/index.js";
export { readTool, bashTool, editTool, writeTool, grepTool, findTool, lsTool, codingTools, readOnlyTools, allTools as allBuiltInTools, createCodingTools, createReadOnlyTools, createReadTool, createBashTool, createEditTool, createWriteTool, createGrepTool, createFindTool, createLsTool, };
/**
 * Create an AuthStorage instance for the given agent directory.
 */
export declare function discoverAuthStorage(agentDir?: string): AuthStorage;
/**
 * Create a ModelRegistry for the given agent directory.
 */
export declare function discoverModels(authStorage: AuthStorage, agentDir?: string): ModelRegistry;
/**
 * Discover extensions from cwd and agentDir.
 * @param eventBus - Shared event bus for extension communication. Pass to createAgentSession too.
 * @param cwd - Current working directory
 * @param agentDir - Agent configuration directory
 */
export declare function discoverExtensions(eventBus: EventBus, cwd?: string, agentDir?: string): Promise<LoadExtensionsResult>;
/**
 * Discover skills from cwd and agentDir.
 */
export declare function discoverSkills(cwd?: string, agentDir?: string, settings?: SkillsSettings): {
    skills: Skill[];
    warnings: SkillWarning[];
};
/**
 * Discover context files (AGENTS.md) walking up from cwd.
 */
export declare function discoverContextFiles(cwd?: string, agentDir?: string): Array<{
    path: string;
    content: string;
}>;
/**
 * Discover prompt templates from cwd and agentDir.
 */
export declare function discoverPromptTemplates(cwd?: string, agentDir?: string): PromptTemplate[];
export interface BuildSystemPromptOptions {
    tools?: Tool[];
    skills?: Skill[];
    contextFiles?: Array<{
        path: string;
        content: string;
    }>;
    cwd?: string;
    appendPrompt?: string;
}
/**
 * Build the default system prompt.
 */
export declare function buildSystemPrompt(options?: BuildSystemPromptOptions): string;
/**
 * Load settings from agentDir/settings.json merged with cwd/.pi/settings.json.
 */
export declare function loadSettings(cwd?: string, agentDir?: string): Settings;
/**
 * Create an AgentSession with the specified options.
 *
 * @example
 * ```typescript
 * // Minimal - uses defaults
 * const { session } = await createAgentSession();
 *
 * // With explicit model
 * import { getModel } from '@mariozechner/pi-ai';
 * const { session } = await createAgentSession({
 *   model: getModel('anthropic', 'claude-opus-4-5'),
 *   thinkingLevel: 'high',
 * });
 *
 * // Continue previous session
 * const { session, modelFallbackMessage } = await createAgentSession({
 *   continueSession: true,
 * });
 *
 * // Full control
 * const { session } = await createAgentSession({
 *   model: myModel,
 *   getApiKey: async () => process.env.MY_KEY,
 *   systemPrompt: 'You are helpful.',
 *   tools: [readTool, bashTool],
 *   skills: [],
 *   sessionManager: SessionManager.inMemory(),
 * });
 * ```
 */
export declare function createAgentSession(options?: CreateAgentSessionOptions): Promise<CreateAgentSessionResult>;
//# sourceMappingURL=sdk.d.ts.map