import { existsSync, mkdirSync, readFileSync, writeFileSync } from "fs";
import { dirname, join } from "path";
import { CONFIG_DIR_NAME, getAgentDir } from "../config.js";
/** Deep merge settings: project/overrides take precedence, nested objects merge recursively */
function deepMergeSettings(base, overrides) {
    const result = { ...base };
    for (const key of Object.keys(overrides)) {
        const overrideValue = overrides[key];
        const baseValue = base[key];
        if (overrideValue === undefined) {
            continue;
        }
        // For nested objects, merge recursively
        if (typeof overrideValue === "object" &&
            overrideValue !== null &&
            !Array.isArray(overrideValue) &&
            typeof baseValue === "object" &&
            baseValue !== null &&
            !Array.isArray(baseValue)) {
            result[key] = { ...baseValue, ...overrideValue };
        }
        else {
            // For primitives and arrays, override value wins
            result[key] = overrideValue;
        }
    }
    return result;
}
export class SettingsManager {
    settingsPath;
    projectSettingsPath;
    globalSettings;
    settings;
    persist;
    constructor(settingsPath, projectSettingsPath, initialSettings, persist) {
        this.settingsPath = settingsPath;
        this.projectSettingsPath = projectSettingsPath;
        this.persist = persist;
        this.globalSettings = initialSettings;
        const projectSettings = this.loadProjectSettings();
        this.settings = deepMergeSettings(this.globalSettings, projectSettings);
    }
    /** Create a SettingsManager that loads from files */
    static create(cwd = process.cwd(), agentDir = getAgentDir()) {
        const settingsPath = join(agentDir, "settings.json");
        const projectSettingsPath = join(cwd, CONFIG_DIR_NAME, "settings.json");
        const globalSettings = SettingsManager.loadFromFile(settingsPath);
        return new SettingsManager(settingsPath, projectSettingsPath, globalSettings, true);
    }
    /** Create an in-memory SettingsManager (no file I/O) */
    static inMemory(settings = {}) {
        return new SettingsManager(null, null, settings, false);
    }
    static loadFromFile(path) {
        if (!existsSync(path)) {
            return {};
        }
        try {
            const content = readFileSync(path, "utf-8");
            const settings = JSON.parse(content);
            return SettingsManager.migrateSettings(settings);
        }
        catch (error) {
            console.error(`Warning: Could not read settings file ${path}: ${error}`);
            return {};
        }
    }
    /** Migrate old settings format to new format */
    static migrateSettings(settings) {
        // Migrate queueMode -> steeringMode
        if ("queueMode" in settings && !("steeringMode" in settings)) {
            settings.steeringMode = settings.queueMode;
            delete settings.queueMode;
        }
        return settings;
    }
    loadProjectSettings() {
        if (!this.projectSettingsPath || !existsSync(this.projectSettingsPath)) {
            return {};
        }
        try {
            const content = readFileSync(this.projectSettingsPath, "utf-8");
            const settings = JSON.parse(content);
            return SettingsManager.migrateSettings(settings);
        }
        catch (error) {
            console.error(`Warning: Could not read project settings file: ${error}`);
            return {};
        }
    }
    /** Apply additional overrides on top of current settings */
    applyOverrides(overrides) {
        this.settings = deepMergeSettings(this.settings, overrides);
    }
    save() {
        if (this.persist && this.settingsPath) {
            try {
                const dir = dirname(this.settingsPath);
                if (!existsSync(dir)) {
                    mkdirSync(dir, { recursive: true });
                }
                // Re-read current file to preserve any settings added externally while running
                const currentFileSettings = SettingsManager.loadFromFile(this.settingsPath);
                // Merge: file settings as base, globalSettings (in-memory changes) as overrides
                const mergedSettings = deepMergeSettings(currentFileSettings, this.globalSettings);
                this.globalSettings = mergedSettings;
                // Save merged settings (project settings are read-only)
                writeFileSync(this.settingsPath, JSON.stringify(this.globalSettings, null, 2), "utf-8");
            }
            catch (error) {
                console.error(`Warning: Could not save settings file: ${error}`);
            }
        }
        // Always re-merge to update active settings (needed for both file and inMemory modes)
        const projectSettings = this.loadProjectSettings();
        this.settings = deepMergeSettings(this.globalSettings, projectSettings);
    }
    getLastChangelogVersion() {
        return this.settings.lastChangelogVersion;
    }
    setLastChangelogVersion(version) {
        this.globalSettings.lastChangelogVersion = version;
        this.save();
    }
    getDefaultProvider() {
        return this.settings.defaultProvider;
    }
    getDefaultModel() {
        return this.settings.defaultModel;
    }
    setDefaultProvider(provider) {
        this.globalSettings.defaultProvider = provider;
        this.save();
    }
    setDefaultModel(modelId) {
        this.globalSettings.defaultModel = modelId;
        this.save();
    }
    setDefaultModelAndProvider(provider, modelId) {
        this.globalSettings.defaultProvider = provider;
        this.globalSettings.defaultModel = modelId;
        this.save();
    }
    getSteeringMode() {
        return this.settings.steeringMode || "one-at-a-time";
    }
    setSteeringMode(mode) {
        this.globalSettings.steeringMode = mode;
        this.save();
    }
    getFollowUpMode() {
        return this.settings.followUpMode || "one-at-a-time";
    }
    setFollowUpMode(mode) {
        this.globalSettings.followUpMode = mode;
        this.save();
    }
    getTheme() {
        return this.settings.theme;
    }
    setTheme(theme) {
        this.globalSettings.theme = theme;
        this.save();
    }
    getDefaultThinkingLevel() {
        return this.settings.defaultThinkingLevel;
    }
    setDefaultThinkingLevel(level) {
        this.globalSettings.defaultThinkingLevel = level;
        this.save();
    }
    getCompactionEnabled() {
        return this.settings.compaction?.enabled ?? true;
    }
    setCompactionEnabled(enabled) {
        if (!this.globalSettings.compaction) {
            this.globalSettings.compaction = {};
        }
        this.globalSettings.compaction.enabled = enabled;
        this.save();
    }
    getCompactionReserveTokens() {
        return this.settings.compaction?.reserveTokens ?? 16384;
    }
    getCompactionKeepRecentTokens() {
        return this.settings.compaction?.keepRecentTokens ?? 20000;
    }
    getCompactionSettings() {
        return {
            enabled: this.getCompactionEnabled(),
            reserveTokens: this.getCompactionReserveTokens(),
            keepRecentTokens: this.getCompactionKeepRecentTokens(),
        };
    }
    getBranchSummarySettings() {
        return {
            reserveTokens: this.settings.branchSummary?.reserveTokens ?? 16384,
        };
    }
    getRetryEnabled() {
        return this.settings.retry?.enabled ?? true;
    }
    setRetryEnabled(enabled) {
        if (!this.globalSettings.retry) {
            this.globalSettings.retry = {};
        }
        this.globalSettings.retry.enabled = enabled;
        this.save();
    }
    getRetrySettings() {
        return {
            enabled: this.getRetryEnabled(),
            maxRetries: this.settings.retry?.maxRetries ?? 3,
            baseDelayMs: this.settings.retry?.baseDelayMs ?? 2000,
        };
    }
    getHideThinkingBlock() {
        return this.settings.hideThinkingBlock ?? false;
    }
    setHideThinkingBlock(hide) {
        this.globalSettings.hideThinkingBlock = hide;
        this.save();
    }
    getShellPath() {
        return this.settings.shellPath;
    }
    setShellPath(path) {
        this.globalSettings.shellPath = path;
        this.save();
    }
    getQuietStartup() {
        return this.settings.quietStartup ?? false;
    }
    setQuietStartup(quiet) {
        this.globalSettings.quietStartup = quiet;
        this.save();
    }
    getShellCommandPrefix() {
        return this.settings.shellCommandPrefix;
    }
    setShellCommandPrefix(prefix) {
        this.globalSettings.shellCommandPrefix = prefix;
        this.save();
    }
    getCollapseChangelog() {
        return this.settings.collapseChangelog ?? false;
    }
    setCollapseChangelog(collapse) {
        this.globalSettings.collapseChangelog = collapse;
        this.save();
    }
    getExtensionPaths() {
        return [...(this.settings.extensions ?? [])];
    }
    setExtensionPaths(paths) {
        this.globalSettings.extensions = paths;
        this.save();
    }
    getSkillsEnabled() {
        return this.settings.skills?.enabled ?? true;
    }
    setSkillsEnabled(enabled) {
        if (!this.globalSettings.skills) {
            this.globalSettings.skills = {};
        }
        this.globalSettings.skills.enabled = enabled;
        this.save();
    }
    getSkillsSettings() {
        return {
            enabled: this.settings.skills?.enabled ?? true,
            enableCodexUser: this.settings.skills?.enableCodexUser ?? true,
            enableClaudeUser: this.settings.skills?.enableClaudeUser ?? true,
            enableClaudeProject: this.settings.skills?.enableClaudeProject ?? true,
            enablePiUser: this.settings.skills?.enablePiUser ?? true,
            enablePiProject: this.settings.skills?.enablePiProject ?? true,
            enableSkillCommands: this.settings.skills?.enableSkillCommands ?? true,
            customDirectories: [...(this.settings.skills?.customDirectories ?? [])],
            ignoredSkills: [...(this.settings.skills?.ignoredSkills ?? [])],
            includeSkills: [...(this.settings.skills?.includeSkills ?? [])],
        };
    }
    getEnableSkillCommands() {
        return this.settings.skills?.enableSkillCommands ?? true;
    }
    setEnableSkillCommands(enabled) {
        if (!this.globalSettings.skills) {
            this.globalSettings.skills = {};
        }
        this.globalSettings.skills.enableSkillCommands = enabled;
        this.save();
    }
    getThinkingBudgets() {
        return this.settings.thinkingBudgets;
    }
    getShowImages() {
        return this.settings.terminal?.showImages ?? true;
    }
    setShowImages(show) {
        if (!this.globalSettings.terminal) {
            this.globalSettings.terminal = {};
        }
        this.globalSettings.terminal.showImages = show;
        this.save();
    }
    getImageAutoResize() {
        return this.settings.images?.autoResize ?? true;
    }
    setImageAutoResize(enabled) {
        if (!this.globalSettings.images) {
            this.globalSettings.images = {};
        }
        this.globalSettings.images.autoResize = enabled;
        this.save();
    }
    getBlockImages() {
        return this.settings.images?.blockImages ?? false;
    }
    setBlockImages(blocked) {
        if (!this.globalSettings.images) {
            this.globalSettings.images = {};
        }
        this.globalSettings.images.blockImages = blocked;
        this.save();
    }
    getEnabledModels() {
        return this.settings.enabledModels;
    }
    setEnabledModels(patterns) {
        this.globalSettings.enabledModels = patterns;
        this.save();
    }
    getDoubleEscapeAction() {
        return this.settings.doubleEscapeAction ?? "tree";
    }
    setDoubleEscapeAction(action) {
        this.globalSettings.doubleEscapeAction = action;
        this.save();
    }
    getShowHardwareCursor() {
        return this.settings.showHardwareCursor ?? process.env.PI_HARDWARE_CURSOR === "1";
    }
    setShowHardwareCursor(enabled) {
        this.globalSettings.showHardwareCursor = enabled;
        this.save();
    }
    getEditorPaddingX() {
        return this.settings.editorPaddingX ?? 0;
    }
    setEditorPaddingX(padding) {
        this.globalSettings.editorPaddingX = Math.max(0, Math.min(3, Math.floor(padding)));
        this.save();
    }
    getCodeBlockIndent() {
        return this.settings.markdown?.codeBlockIndent ?? "  ";
    }
}
//# sourceMappingURL=settings-manager.js.map