/**
 * Shared diff computation utilities for the edit tool.
 * Used by both edit.ts (for execution) and tool-execution.ts (for preview rendering).
 */
import * as Diff from "diff";
import { constants } from "fs";
import { access, readFile } from "fs/promises";
import { resolveToCwd } from "./path-utils.js";
export function detectLineEnding(content) {
    const crlfIdx = content.indexOf("\r\n");
    const lfIdx = content.indexOf("\n");
    if (lfIdx === -1)
        return "\n";
    if (crlfIdx === -1)
        return "\n";
    return crlfIdx < lfIdx ? "\r\n" : "\n";
}
export function normalizeToLF(text) {
    return text.replace(/\r\n/g, "\n").replace(/\r/g, "\n");
}
export function restoreLineEndings(text, ending) {
    return ending === "\r\n" ? text.replace(/\n/g, "\r\n") : text;
}
/**
 * Normalize text for fuzzy matching. Applies progressive transformations:
 * - Strip trailing whitespace from each line
 * - Normalize smart quotes to ASCII equivalents
 * - Normalize Unicode dashes/hyphens to ASCII hyphen
 * - Normalize special Unicode spaces to regular space
 */
export function normalizeForFuzzyMatch(text) {
    return (text
        // Strip trailing whitespace per line
        .split("\n")
        .map((line) => line.trimEnd())
        .join("\n")
        // Smart single quotes → '
        .replace(/[\u2018\u2019\u201A\u201B]/g, "'")
        // Smart double quotes → "
        .replace(/[\u201C\u201D\u201E\u201F]/g, '"')
        // Various dashes/hyphens → -
        // U+2010 hyphen, U+2011 non-breaking hyphen, U+2012 figure dash,
        // U+2013 en-dash, U+2014 em-dash, U+2015 horizontal bar, U+2212 minus
        .replace(/[\u2010\u2011\u2012\u2013\u2014\u2015\u2212]/g, "-")
        // Special spaces → regular space
        // U+00A0 NBSP, U+2002-U+200A various spaces, U+202F narrow NBSP,
        // U+205F medium math space, U+3000 ideographic space
        .replace(/[\u00A0\u2002-\u200A\u202F\u205F\u3000]/g, " "));
}
/**
 * Find oldText in content, trying exact match first, then fuzzy match.
 * When fuzzy matching is used, the returned contentForReplacement is the
 * fuzzy-normalized version of the content (trailing whitespace stripped,
 * Unicode quotes/dashes normalized to ASCII).
 */
export function fuzzyFindText(content, oldText) {
    // Try exact match first
    const exactIndex = content.indexOf(oldText);
    if (exactIndex !== -1) {
        return {
            found: true,
            index: exactIndex,
            matchLength: oldText.length,
            usedFuzzyMatch: false,
            contentForReplacement: content,
        };
    }
    // Try fuzzy match - work entirely in normalized space
    const fuzzyContent = normalizeForFuzzyMatch(content);
    const fuzzyOldText = normalizeForFuzzyMatch(oldText);
    const fuzzyIndex = fuzzyContent.indexOf(fuzzyOldText);
    if (fuzzyIndex === -1) {
        return {
            found: false,
            index: -1,
            matchLength: 0,
            usedFuzzyMatch: false,
            contentForReplacement: content,
        };
    }
    // When fuzzy matching, we work in the normalized space for replacement.
    // This means the output will have normalized whitespace/quotes/dashes,
    // which is acceptable since we're fixing minor formatting differences anyway.
    return {
        found: true,
        index: fuzzyIndex,
        matchLength: fuzzyOldText.length,
        usedFuzzyMatch: true,
        contentForReplacement: fuzzyContent,
    };
}
/** Strip UTF-8 BOM if present, return both the BOM (if any) and the text without it */
export function stripBom(content) {
    return content.startsWith("\uFEFF") ? { bom: "\uFEFF", text: content.slice(1) } : { bom: "", text: content };
}
/**
 * Generate a unified diff string with line numbers and context.
 * Returns both the diff string and the first changed line number (in the new file).
 */
export function generateDiffString(oldContent, newContent, contextLines = 4) {
    const parts = Diff.diffLines(oldContent, newContent);
    const output = [];
    const oldLines = oldContent.split("\n");
    const newLines = newContent.split("\n");
    const maxLineNum = Math.max(oldLines.length, newLines.length);
    const lineNumWidth = String(maxLineNum).length;
    let oldLineNum = 1;
    let newLineNum = 1;
    let lastWasChange = false;
    let firstChangedLine;
    for (let i = 0; i < parts.length; i++) {
        const part = parts[i];
        const raw = part.value.split("\n");
        if (raw[raw.length - 1] === "") {
            raw.pop();
        }
        if (part.added || part.removed) {
            // Capture the first changed line (in the new file)
            if (firstChangedLine === undefined) {
                firstChangedLine = newLineNum;
            }
            // Show the change
            for (const line of raw) {
                if (part.added) {
                    const lineNum = String(newLineNum).padStart(lineNumWidth, " ");
                    output.push(`+${lineNum} ${line}`);
                    newLineNum++;
                }
                else {
                    // removed
                    const lineNum = String(oldLineNum).padStart(lineNumWidth, " ");
                    output.push(`-${lineNum} ${line}`);
                    oldLineNum++;
                }
            }
            lastWasChange = true;
        }
        else {
            // Context lines - only show a few before/after changes
            const nextPartIsChange = i < parts.length - 1 && (parts[i + 1].added || parts[i + 1].removed);
            if (lastWasChange || nextPartIsChange) {
                // Show context
                let linesToShow = raw;
                let skipStart = 0;
                let skipEnd = 0;
                if (!lastWasChange) {
                    // Show only last N lines as leading context
                    skipStart = Math.max(0, raw.length - contextLines);
                    linesToShow = raw.slice(skipStart);
                }
                if (!nextPartIsChange && linesToShow.length > contextLines) {
                    // Show only first N lines as trailing context
                    skipEnd = linesToShow.length - contextLines;
                    linesToShow = linesToShow.slice(0, contextLines);
                }
                // Add ellipsis if we skipped lines at start
                if (skipStart > 0) {
                    output.push(` ${"".padStart(lineNumWidth, " ")} ...`);
                    // Update line numbers for the skipped leading context
                    oldLineNum += skipStart;
                    newLineNum += skipStart;
                }
                for (const line of linesToShow) {
                    const lineNum = String(oldLineNum).padStart(lineNumWidth, " ");
                    output.push(` ${lineNum} ${line}`);
                    oldLineNum++;
                    newLineNum++;
                }
                // Add ellipsis if we skipped lines at end
                if (skipEnd > 0) {
                    output.push(` ${"".padStart(lineNumWidth, " ")} ...`);
                    // Update line numbers for the skipped trailing context
                    oldLineNum += skipEnd;
                    newLineNum += skipEnd;
                }
            }
            else {
                // Skip these context lines entirely
                oldLineNum += raw.length;
                newLineNum += raw.length;
            }
            lastWasChange = false;
        }
    }
    return { diff: output.join("\n"), firstChangedLine };
}
/**
 * Compute the diff for an edit operation without applying it.
 * Used for preview rendering in the TUI before the tool executes.
 */
export async function computeEditDiff(path, oldText, newText, cwd) {
    const absolutePath = resolveToCwd(path, cwd);
    try {
        // Check if file exists and is readable
        try {
            await access(absolutePath, constants.R_OK);
        }
        catch {
            return { error: `File not found: ${path}` };
        }
        // Read the file
        const rawContent = await readFile(absolutePath, "utf-8");
        // Strip BOM before matching (LLM won't include invisible BOM in oldText)
        const { text: content } = stripBom(rawContent);
        const normalizedContent = normalizeToLF(content);
        const normalizedOldText = normalizeToLF(oldText);
        const normalizedNewText = normalizeToLF(newText);
        // Find the old text using fuzzy matching (tries exact match first, then fuzzy)
        const matchResult = fuzzyFindText(normalizedContent, normalizedOldText);
        if (!matchResult.found) {
            return {
                error: `Could not find the exact text in ${path}. The old text must match exactly including all whitespace and newlines.`,
            };
        }
        // Count occurrences using fuzzy-normalized content for consistency
        const fuzzyContent = normalizeForFuzzyMatch(normalizedContent);
        const fuzzyOldText = normalizeForFuzzyMatch(normalizedOldText);
        const occurrences = fuzzyContent.split(fuzzyOldText).length - 1;
        if (occurrences > 1) {
            return {
                error: `Found ${occurrences} occurrences of the text in ${path}. The text must be unique. Please provide more context to make it unique.`,
            };
        }
        // Compute the new content using the matched position
        // When fuzzy matching was used, contentForReplacement is the normalized version
        const baseContent = matchResult.contentForReplacement;
        const newContent = baseContent.substring(0, matchResult.index) +
            normalizedNewText +
            baseContent.substring(matchResult.index + matchResult.matchLength);
        // Check if it would actually change anything
        if (baseContent === newContent) {
            return {
                error: `No changes would be made to ${path}. The replacement produces identical content.`,
            };
        }
        // Generate the diff
        return generateDiffString(baseContent, newContent);
    }
    catch (err) {
        return { error: err instanceof Error ? err.message : String(err) };
    }
}
//# sourceMappingURL=edit-diff.js.map