import { Type } from "@sinclair/typebox";
import { existsSync, readdirSync, statSync } from "fs";
import nodePath from "path";
import { resolveToCwd } from "./path-utils.js";
import { DEFAULT_MAX_BYTES, formatSize, truncateHead } from "./truncate.js";
const lsSchema = Type.Object({
    path: Type.Optional(Type.String({ description: "Directory to list (default: current directory)" })),
    limit: Type.Optional(Type.Number({ description: "Maximum number of entries to return (default: 500)" })),
});
const DEFAULT_LIMIT = 500;
const defaultLsOperations = {
    exists: existsSync,
    stat: statSync,
    readdir: readdirSync,
};
export function createLsTool(cwd, options) {
    const ops = options?.operations ?? defaultLsOperations;
    return {
        name: "ls",
        label: "ls",
        description: `List directory contents. Returns entries sorted alphabetically, with '/' suffix for directories. Includes dotfiles. Output is truncated to ${DEFAULT_LIMIT} entries or ${DEFAULT_MAX_BYTES / 1024}KB (whichever is hit first).`,
        parameters: lsSchema,
        execute: async (_toolCallId, { path, limit }, signal) => {
            return new Promise((resolve, reject) => {
                if (signal?.aborted) {
                    reject(new Error("Operation aborted"));
                    return;
                }
                const onAbort = () => reject(new Error("Operation aborted"));
                signal?.addEventListener("abort", onAbort, { once: true });
                (async () => {
                    try {
                        const dirPath = resolveToCwd(path || ".", cwd);
                        const effectiveLimit = limit ?? DEFAULT_LIMIT;
                        // Check if path exists
                        if (!(await ops.exists(dirPath))) {
                            reject(new Error(`Path not found: ${dirPath}`));
                            return;
                        }
                        // Check if path is a directory
                        const stat = await ops.stat(dirPath);
                        if (!stat.isDirectory()) {
                            reject(new Error(`Not a directory: ${dirPath}`));
                            return;
                        }
                        // Read directory entries
                        let entries;
                        try {
                            entries = await ops.readdir(dirPath);
                        }
                        catch (e) {
                            reject(new Error(`Cannot read directory: ${e.message}`));
                            return;
                        }
                        // Sort alphabetically (case-insensitive)
                        entries.sort((a, b) => a.toLowerCase().localeCompare(b.toLowerCase()));
                        // Format entries with directory indicators
                        const results = [];
                        let entryLimitReached = false;
                        for (const entry of entries) {
                            if (results.length >= effectiveLimit) {
                                entryLimitReached = true;
                                break;
                            }
                            const fullPath = nodePath.join(dirPath, entry);
                            let suffix = "";
                            try {
                                const entryStat = await ops.stat(fullPath);
                                if (entryStat.isDirectory()) {
                                    suffix = "/";
                                }
                            }
                            catch {
                                // Skip entries we can't stat
                                continue;
                            }
                            results.push(entry + suffix);
                        }
                        signal?.removeEventListener("abort", onAbort);
                        if (results.length === 0) {
                            resolve({ content: [{ type: "text", text: "(empty directory)" }], details: undefined });
                            return;
                        }
                        // Apply byte truncation (no line limit since we already have entry limit)
                        const rawOutput = results.join("\n");
                        const truncation = truncateHead(rawOutput, { maxLines: Number.MAX_SAFE_INTEGER });
                        let output = truncation.content;
                        const details = {};
                        // Build notices
                        const notices = [];
                        if (entryLimitReached) {
                            notices.push(`${effectiveLimit} entries limit reached. Use limit=${effectiveLimit * 2} for more`);
                            details.entryLimitReached = effectiveLimit;
                        }
                        if (truncation.truncated) {
                            notices.push(`${formatSize(DEFAULT_MAX_BYTES)} limit reached`);
                            details.truncation = truncation;
                        }
                        if (notices.length > 0) {
                            output += `\n\n[${notices.join(". ")}]`;
                        }
                        resolve({
                            content: [{ type: "text", text: output }],
                            details: Object.keys(details).length > 0 ? details : undefined,
                        });
                    }
                    catch (e) {
                        signal?.removeEventListener("abort", onAbort);
                        reject(e);
                    }
                })();
            });
        },
    };
}
/** Default ls tool using process.cwd() - for backwards compatibility */
export const lsTool = createLsTool(process.cwd());
//# sourceMappingURL=ls.js.map