import { spawnSync } from "node:child_process";
import { existsSync } from "node:fs";
import { unlink } from "node:fs/promises";
import * as os from "node:os";
import { Container, getEditorKeybindings, Input, matchesKey, Spacer, truncateToWidth, visibleWidth, } from "@mariozechner/pi-tui";
import { theme } from "../theme/theme.js";
import { DynamicBorder } from "./dynamic-border.js";
import { keyHint, rawKeyHint } from "./keybinding-hints.js";
import { filterAndSortSessions } from "./session-selector-search.js";
function shortenPath(path) {
    const home = os.homedir();
    if (!path)
        return path;
    if (path.startsWith(home)) {
        return `~${path.slice(home.length)}`;
    }
    return path;
}
function formatSessionDate(date) {
    const now = new Date();
    const diffMs = now.getTime() - date.getTime();
    const diffMins = Math.floor(diffMs / 60000);
    const diffHours = Math.floor(diffMs / 3600000);
    const diffDays = Math.floor(diffMs / 86400000);
    if (diffMins < 1)
        return "just now";
    if (diffMins < 60)
        return `${diffMins} minute${diffMins !== 1 ? "s" : ""} ago`;
    if (diffHours < 24)
        return `${diffHours} hour${diffHours !== 1 ? "s" : ""} ago`;
    if (diffDays === 1)
        return "1 day ago";
    if (diffDays < 7)
        return `${diffDays} days ago`;
    return date.toLocaleDateString();
}
class SessionSelectorHeader {
    scope;
    sortMode;
    requestRender;
    loading = false;
    loadProgress = null;
    showPath = false;
    confirmingDeletePath = null;
    statusMessage = null;
    statusTimeout = null;
    constructor(scope, sortMode, requestRender) {
        this.scope = scope;
        this.sortMode = sortMode;
        this.requestRender = requestRender;
    }
    setScope(scope) {
        this.scope = scope;
    }
    setSortMode(sortMode) {
        this.sortMode = sortMode;
    }
    setLoading(loading) {
        this.loading = loading;
        // Progress is scoped to the current load; clear whenever the loading state is set
        this.loadProgress = null;
    }
    setProgress(loaded, total) {
        this.loadProgress = { loaded, total };
    }
    setShowPath(showPath) {
        this.showPath = showPath;
    }
    setConfirmingDeletePath(path) {
        this.confirmingDeletePath = path;
    }
    clearStatusTimeout() {
        if (!this.statusTimeout)
            return;
        clearTimeout(this.statusTimeout);
        this.statusTimeout = null;
    }
    setStatusMessage(msg, autoHideMs) {
        this.clearStatusTimeout();
        this.statusMessage = msg;
        if (!msg || !autoHideMs)
            return;
        this.statusTimeout = setTimeout(() => {
            this.statusMessage = null;
            this.statusTimeout = null;
            this.requestRender();
        }, autoHideMs);
    }
    invalidate() { }
    render(width) {
        const title = this.scope === "current" ? "Resume Session (Current Folder)" : "Resume Session (All)";
        const leftText = theme.bold(title);
        const sortLabel = this.sortMode === "recent" ? "Recent" : "Fuzzy";
        const sortText = theme.fg("muted", "Sort: ") + theme.fg("accent", sortLabel);
        let scopeText;
        if (this.loading) {
            const progressText = this.loadProgress ? `${this.loadProgress.loaded}/${this.loadProgress.total}` : "...";
            scopeText = `${theme.fg("muted", "○ Current Folder | ")}${theme.fg("accent", `Loading ${progressText}`)}`;
        }
        else if (this.scope === "current") {
            scopeText = `${theme.fg("accent", "◉ Current Folder")}${theme.fg("muted", " | ○ All")}`;
        }
        else {
            scopeText = `${theme.fg("muted", "○ Current Folder | ")}${theme.fg("accent", "◉ All")}`;
        }
        const rightText = truncateToWidth(`${scopeText}  ${sortText}`, width, "");
        const availableLeft = Math.max(0, width - visibleWidth(rightText) - 1);
        const left = truncateToWidth(leftText, availableLeft, "");
        const spacing = Math.max(0, width - visibleWidth(left) - visibleWidth(rightText));
        // Build hint lines - changes based on state (all branches truncate to width)
        let hintLine1;
        let hintLine2;
        if (this.confirmingDeletePath !== null) {
            const confirmHint = "Delete session? [Enter] confirm · [Esc/Ctrl+C] cancel";
            hintLine1 = theme.fg("error", truncateToWidth(confirmHint, width, "…"));
            hintLine2 = "";
        }
        else if (this.statusMessage) {
            const color = this.statusMessage.type === "error" ? "error" : "accent";
            hintLine1 = theme.fg(color, truncateToWidth(this.statusMessage.message, width, "…"));
            hintLine2 = "";
        }
        else {
            const pathState = this.showPath ? "(on)" : "(off)";
            const sep = theme.fg("muted", " · ");
            const hint1 = keyHint("tab", "scope") + sep + theme.fg("muted", 're:<pattern> regex · "phrase" exact');
            const hint2 = rawKeyHint("ctrl+r", "sort") +
                sep +
                rawKeyHint("ctrl+d", "delete") +
                sep +
                rawKeyHint("ctrl+p", `path ${pathState}`);
            hintLine1 = truncateToWidth(hint1, width, "…");
            hintLine2 = truncateToWidth(hint2, width, "…");
        }
        return [`${left}${" ".repeat(spacing)}${rightText}`, hintLine1, hintLine2];
    }
}
/**
 * Custom session list component with multi-line items and search
 */
class SessionList {
    allSessions = [];
    filteredSessions = [];
    selectedIndex = 0;
    searchInput;
    showCwd = false;
    sortMode = "relevance";
    showPath = false;
    confirmingDeletePath = null;
    currentSessionFilePath;
    onSelect;
    onCancel;
    onExit = () => { };
    onToggleScope;
    onToggleSort;
    onTogglePath;
    onDeleteConfirmationChange;
    onDeleteSession;
    onError;
    maxVisible = 5; // Max sessions visible (each session: message + metadata + optional path + blank)
    // Focusable implementation - propagate to searchInput for IME cursor positioning
    _focused = false;
    get focused() {
        return this._focused;
    }
    set focused(value) {
        this._focused = value;
        this.searchInput.focused = value;
    }
    constructor(sessions, showCwd, sortMode, currentSessionFilePath) {
        this.allSessions = sessions;
        this.filteredSessions = sessions;
        this.searchInput = new Input();
        this.showCwd = showCwd;
        this.sortMode = sortMode;
        this.currentSessionFilePath = currentSessionFilePath;
        // Handle Enter in search input - select current item
        this.searchInput.onSubmit = () => {
            if (this.filteredSessions[this.selectedIndex]) {
                const selected = this.filteredSessions[this.selectedIndex];
                if (this.onSelect) {
                    this.onSelect(selected.path);
                }
            }
        };
    }
    setSortMode(sortMode) {
        this.sortMode = sortMode;
        this.filterSessions(this.searchInput.getValue());
    }
    setSessions(sessions, showCwd) {
        this.allSessions = sessions;
        this.showCwd = showCwd;
        this.filterSessions(this.searchInput.getValue());
    }
    filterSessions(query) {
        this.filteredSessions = filterAndSortSessions(this.allSessions, query, this.sortMode);
        this.selectedIndex = Math.min(this.selectedIndex, Math.max(0, this.filteredSessions.length - 1));
    }
    setConfirmingDeletePath(path) {
        this.confirmingDeletePath = path;
        this.onDeleteConfirmationChange?.(path);
    }
    startDeleteConfirmationForSelectedSession() {
        const selected = this.filteredSessions[this.selectedIndex];
        if (!selected)
            return;
        // Prevent deleting current session
        if (this.currentSessionFilePath && selected.path === this.currentSessionFilePath) {
            this.onError?.("Cannot delete the currently active session");
            return;
        }
        this.setConfirmingDeletePath(selected.path);
    }
    invalidate() { }
    render(width) {
        const lines = [];
        // Render search input
        lines.push(...this.searchInput.render(width));
        lines.push(""); // Blank line after search
        if (this.filteredSessions.length === 0) {
            if (this.showCwd) {
                // "All" scope - no sessions anywhere that match filter
                lines.push(theme.fg("muted", truncateToWidth("  No sessions found", width, "…")));
            }
            else {
                // "Current folder" scope - hint to try "all"
                lines.push(theme.fg("muted", truncateToWidth("  No sessions in current folder. Press Tab to view all.", width, "…")));
            }
            return lines;
        }
        // Calculate visible range with scrolling
        const startIndex = Math.max(0, Math.min(this.selectedIndex - Math.floor(this.maxVisible / 2), this.filteredSessions.length - this.maxVisible));
        const endIndex = Math.min(startIndex + this.maxVisible, this.filteredSessions.length);
        // Render visible sessions (message + metadata + optional path + blank line)
        for (let i = startIndex; i < endIndex; i++) {
            const session = this.filteredSessions[i];
            const isSelected = i === this.selectedIndex;
            const isConfirmingDelete = session.path === this.confirmingDeletePath;
            // Use session name if set, otherwise first message
            const hasName = !!session.name;
            const displayText = session.name ?? session.firstMessage;
            const normalizedMessage = displayText.replace(/\n/g, " ").trim();
            // First line: cursor + message (truncate to visible width)
            // Use warning color for custom names to distinguish from first message
            const cursor = isSelected ? theme.fg("accent", "› ") : "  ";
            const maxMsgWidth = width - 2; // Account for cursor (2 visible chars)
            const truncatedMsg = truncateToWidth(normalizedMessage, maxMsgWidth, "...");
            let messageColor = null;
            if (isConfirmingDelete) {
                messageColor = "error";
            }
            else if (hasName) {
                messageColor = "warning";
            }
            let styledMsg = messageColor ? theme.fg(messageColor, truncatedMsg) : truncatedMsg;
            if (isSelected) {
                styledMsg = theme.bold(styledMsg);
            }
            const messageLine = cursor + styledMsg;
            // Second line: metadata (dimmed) - also truncate for safety
            const modified = formatSessionDate(session.modified);
            const msgCount = `${session.messageCount} message${session.messageCount !== 1 ? "s" : ""}`;
            const metadataParts = [modified, msgCount];
            if (this.showCwd && session.cwd) {
                metadataParts.push(shortenPath(session.cwd));
            }
            const metadata = `  ${metadataParts.join(" · ")}`;
            const truncatedMetadata = truncateToWidth(metadata, width, "");
            const metadataLine = theme.fg(isConfirmingDelete ? "error" : "dim", truncatedMetadata);
            lines.push(messageLine);
            lines.push(metadataLine);
            // Optional third line: file path (when showPath is enabled)
            if (this.showPath) {
                const pathText = `  ${shortenPath(session.path)}`;
                const truncatedPath = truncateToWidth(pathText, width, "…");
                const pathLine = theme.fg(isConfirmingDelete ? "error" : "muted", truncatedPath);
                lines.push(pathLine);
            }
            lines.push(""); // Blank line between sessions
        }
        // Add scroll indicator if needed
        if (startIndex > 0 || endIndex < this.filteredSessions.length) {
            const scrollText = `  (${this.selectedIndex + 1}/${this.filteredSessions.length})`;
            const scrollInfo = theme.fg("muted", truncateToWidth(scrollText, width, ""));
            lines.push(scrollInfo);
        }
        return lines;
    }
    handleInput(keyData) {
        const kb = getEditorKeybindings();
        // Handle delete confirmation state first - intercept all keys
        if (this.confirmingDeletePath !== null) {
            if (kb.matches(keyData, "selectConfirm")) {
                const pathToDelete = this.confirmingDeletePath;
                this.setConfirmingDeletePath(null);
                void this.onDeleteSession?.(pathToDelete);
                return;
            }
            // Allow both Escape and Ctrl+C to cancel (consistent with pi UX)
            if (kb.matches(keyData, "selectCancel") || matchesKey(keyData, "ctrl+c")) {
                this.setConfirmingDeletePath(null);
                return;
            }
            // Ignore all other keys while confirming
            return;
        }
        if (kb.matches(keyData, "tab")) {
            if (this.onToggleScope) {
                this.onToggleScope();
            }
            return;
        }
        if (matchesKey(keyData, "ctrl+r")) {
            this.onToggleSort?.();
            return;
        }
        // Ctrl+P: toggle path display
        if (matchesKey(keyData, "ctrl+p")) {
            this.showPath = !this.showPath;
            this.onTogglePath?.(this.showPath);
            return;
        }
        // Ctrl+D: initiate delete confirmation (useful on terminals that don't distinguish Ctrl+Backspace from Backspace)
        if (matchesKey(keyData, "ctrl+d")) {
            this.startDeleteConfirmationForSelectedSession();
            return;
        }
        // Ctrl+Backspace: non-invasive convenience alias for delete
        // Only triggers deletion when the query is empty; otherwise it is forwarded to the input
        if (matchesKey(keyData, "ctrl+backspace")) {
            if (this.searchInput.getValue().length > 0) {
                this.searchInput.handleInput(keyData);
                this.filterSessions(this.searchInput.getValue());
                return;
            }
            this.startDeleteConfirmationForSelectedSession();
            return;
        }
        // Up arrow
        if (kb.matches(keyData, "selectUp")) {
            this.selectedIndex = Math.max(0, this.selectedIndex - 1);
        }
        // Down arrow
        else if (kb.matches(keyData, "selectDown")) {
            this.selectedIndex = Math.min(this.filteredSessions.length - 1, this.selectedIndex + 1);
        }
        // Page up - jump up by maxVisible items
        else if (kb.matches(keyData, "selectPageUp")) {
            this.selectedIndex = Math.max(0, this.selectedIndex - this.maxVisible);
        }
        // Page down - jump down by maxVisible items
        else if (kb.matches(keyData, "selectPageDown")) {
            this.selectedIndex = Math.min(this.filteredSessions.length - 1, this.selectedIndex + this.maxVisible);
        }
        // Enter
        else if (kb.matches(keyData, "selectConfirm")) {
            const selected = this.filteredSessions[this.selectedIndex];
            if (selected && this.onSelect) {
                this.onSelect(selected.path);
            }
        }
        // Escape - cancel
        else if (kb.matches(keyData, "selectCancel")) {
            if (this.onCancel) {
                this.onCancel();
            }
        }
        // Pass everything else to search input
        else {
            this.searchInput.handleInput(keyData);
            this.filterSessions(this.searchInput.getValue());
        }
    }
}
/**
 * Delete a session file, trying the `trash` CLI first, then falling back to unlink
 */
async function deleteSessionFile(sessionPath) {
    // Try `trash` first (if installed)
    const trashArgs = sessionPath.startsWith("-") ? ["--", sessionPath] : [sessionPath];
    const trashResult = spawnSync("trash", trashArgs, { encoding: "utf-8" });
    const getTrashErrorHint = () => {
        const parts = [];
        if (trashResult.error) {
            parts.push(trashResult.error.message);
        }
        const stderr = trashResult.stderr?.trim();
        if (stderr) {
            parts.push(stderr.split("\n")[0] ?? stderr);
        }
        if (parts.length === 0)
            return null;
        return `trash: ${parts.join(" · ").slice(0, 200)}`;
    };
    // If trash reports success, or the file is gone afterwards, treat it as successful
    if (trashResult.status === 0 || !existsSync(sessionPath)) {
        return { ok: true, method: "trash" };
    }
    // Fallback to permanent deletion
    try {
        await unlink(sessionPath);
        return { ok: true, method: "unlink" };
    }
    catch (err) {
        const unlinkError = err instanceof Error ? err.message : String(err);
        const trashErrorHint = getTrashErrorHint();
        const error = trashErrorHint ? `${unlinkError} (${trashErrorHint})` : unlinkError;
        return { ok: false, method: "unlink", error };
    }
}
/**
 * Component that renders a session selector
 */
export class SessionSelectorComponent extends Container {
    sessionList;
    header;
    scope = "current";
    sortMode = "relevance";
    currentSessions = null;
    allSessions = null;
    currentSessionsLoader;
    allSessionsLoader;
    onCancel;
    requestRender;
    currentLoading = false;
    allLoading = false;
    allLoadSeq = 0;
    // Focusable implementation - propagate to sessionList for IME cursor positioning
    _focused = false;
    get focused() {
        return this._focused;
    }
    set focused(value) {
        this._focused = value;
        this.sessionList.focused = value;
    }
    constructor(currentSessionsLoader, allSessionsLoader, onSelect, onCancel, onExit, requestRender, currentSessionFilePath) {
        super();
        this.currentSessionsLoader = currentSessionsLoader;
        this.allSessionsLoader = allSessionsLoader;
        this.onCancel = onCancel;
        this.requestRender = requestRender;
        this.header = new SessionSelectorHeader(this.scope, this.sortMode, this.requestRender);
        // Add header
        this.addChild(new Spacer(1));
        this.addChild(new DynamicBorder());
        this.addChild(new Spacer(1));
        this.addChild(this.header);
        this.addChild(new Spacer(1));
        // Create session list (starts empty, will be populated after load)
        this.sessionList = new SessionList([], false, this.sortMode, currentSessionFilePath);
        // Ensure header status timeouts are cleared when leaving the selector
        const clearStatusMessage = () => this.header.setStatusMessage(null);
        this.sessionList.onSelect = (sessionPath) => {
            clearStatusMessage();
            onSelect(sessionPath);
        };
        this.sessionList.onCancel = () => {
            clearStatusMessage();
            onCancel();
        };
        this.sessionList.onExit = () => {
            clearStatusMessage();
            onExit();
        };
        this.sessionList.onToggleScope = () => this.toggleScope();
        this.sessionList.onToggleSort = () => this.toggleSortMode();
        // Sync list events to header
        this.sessionList.onTogglePath = (showPath) => {
            this.header.setShowPath(showPath);
            this.requestRender();
        };
        this.sessionList.onDeleteConfirmationChange = (path) => {
            this.header.setConfirmingDeletePath(path);
            this.requestRender();
        };
        this.sessionList.onError = (msg) => {
            this.header.setStatusMessage({ type: "error", message: msg }, 3000);
            this.requestRender();
        };
        // Handle session deletion
        this.sessionList.onDeleteSession = async (sessionPath) => {
            const result = await deleteSessionFile(sessionPath);
            if (result.ok) {
                if (this.currentSessions) {
                    this.currentSessions = this.currentSessions.filter((s) => s.path !== sessionPath);
                }
                if (this.allSessions) {
                    this.allSessions = this.allSessions.filter((s) => s.path !== sessionPath);
                }
                const sessions = this.scope === "all" ? (this.allSessions ?? []) : (this.currentSessions ?? []);
                const showCwd = this.scope === "all";
                this.sessionList.setSessions(sessions, showCwd);
                const msg = result.method === "trash" ? "Session moved to trash" : "Session deleted";
                this.header.setStatusMessage({ type: "info", message: msg }, 2000);
            }
            else {
                const errorMessage = result.error ?? "Unknown error";
                this.header.setStatusMessage({ type: "error", message: `Failed to delete: ${errorMessage}` }, 3000);
            }
            this.requestRender();
        };
        this.addChild(this.sessionList);
        // Add bottom border
        this.addChild(new Spacer(1));
        this.addChild(new DynamicBorder());
        // Start loading current sessions immediately
        this.loadCurrentSessions();
    }
    loadCurrentSessions() {
        this.currentLoading = true;
        this.header.setScope("current");
        this.header.setLoading(true);
        this.requestRender();
        this.currentSessionsLoader((loaded, total) => {
            if (this.scope !== "current")
                return;
            this.header.setProgress(loaded, total);
            this.requestRender();
        })
            .then((sessions) => {
            this.currentSessions = sessions;
            this.currentLoading = false;
            if (this.scope !== "current")
                return;
            this.header.setLoading(false);
            this.sessionList.setSessions(sessions, false);
            this.requestRender();
        })
            .catch((error) => {
            this.currentLoading = false;
            const message = error instanceof Error ? error.message : String(error);
            if (this.scope !== "current")
                return;
            this.header.setLoading(false);
            this.header.setStatusMessage({ type: "error", message: `Failed to load sessions: ${message}` }, 4000);
            this.sessionList.setSessions([], false);
            this.requestRender();
        });
    }
    toggleSortMode() {
        this.sortMode = this.sortMode === "recent" ? "relevance" : "recent";
        this.header.setSortMode(this.sortMode);
        this.sessionList.setSortMode(this.sortMode);
        this.requestRender();
    }
    toggleScope() {
        if (this.scope === "current") {
            this.scope = "all";
            this.header.setScope(this.scope);
            if (this.allSessions !== null) {
                this.header.setLoading(false);
                this.sessionList.setSessions(this.allSessions, true);
                this.requestRender();
                return;
            }
            this.header.setLoading(true);
            this.sessionList.setSessions([], true);
            this.requestRender();
            if (this.allLoading)
                return;
            this.allLoading = true;
            const seq = ++this.allLoadSeq;
            this.allSessionsLoader((loaded, total) => {
                if (seq !== this.allLoadSeq)
                    return;
                if (this.scope !== "all")
                    return;
                this.header.setProgress(loaded, total);
                this.requestRender();
            })
                .then((sessions) => {
                this.allSessions = sessions;
                this.allLoading = false;
                if (seq !== this.allLoadSeq)
                    return;
                if (this.scope !== "all")
                    return;
                this.header.setLoading(false);
                this.sessionList.setSessions(sessions, true);
                this.requestRender();
                if (sessions.length === 0 && (this.currentSessions?.length ?? 0) === 0) {
                    this.onCancel();
                }
            })
                .catch((error) => {
                this.allLoading = false;
                const message = error instanceof Error ? error.message : String(error);
                if (seq !== this.allLoadSeq)
                    return;
                if (this.scope !== "all")
                    return;
                this.header.setLoading(false);
                this.header.setStatusMessage({ type: "error", message: `Failed to load sessions: ${message}` }, 4000);
                this.sessionList.setSessions([], true);
                this.requestRender();
            });
        }
        else {
            this.scope = "current";
            this.header.setScope(this.scope);
            this.header.setLoading(this.currentLoading);
            this.sessionList.setSessions(this.currentSessions ?? [], false);
            this.requestRender();
        }
    }
    getSessionList() {
        return this.sessionList;
    }
}
//# sourceMappingURL=session-selector.js.map