import { Container, getEditorKeybindings, Spacer, Text, truncateToWidth } from "@mariozechner/pi-tui";
import { theme } from "../theme/theme.js";
import { DynamicBorder } from "./dynamic-border.js";
/**
 * Custom user message list component with selection
 */
class UserMessageList {
    messages = [];
    selectedIndex = 0;
    onSelect;
    onCancel;
    maxVisible = 10; // Max messages visible
    constructor(messages) {
        // Store messages in chronological order (oldest to newest)
        this.messages = messages;
        // Start with the last (most recent) message selected
        this.selectedIndex = Math.max(0, messages.length - 1);
    }
    invalidate() {
        // No cached state to invalidate currently
    }
    render(width) {
        const lines = [];
        if (this.messages.length === 0) {
            lines.push(theme.fg("muted", "  No user messages found"));
            return lines;
        }
        // Calculate visible range with scrolling
        const startIndex = Math.max(0, Math.min(this.selectedIndex - Math.floor(this.maxVisible / 2), this.messages.length - this.maxVisible));
        const endIndex = Math.min(startIndex + this.maxVisible, this.messages.length);
        // Render visible messages (2 lines per message + blank line)
        for (let i = startIndex; i < endIndex; i++) {
            const message = this.messages[i];
            const isSelected = i === this.selectedIndex;
            // Normalize message to single line
            const normalizedMessage = message.text.replace(/\n/g, " ").trim();
            // First line: cursor + message
            const cursor = isSelected ? theme.fg("accent", "› ") : "  ";
            const maxMsgWidth = width - 2; // Account for cursor (2 chars)
            const truncatedMsg = truncateToWidth(normalizedMessage, maxMsgWidth);
            const messageLine = cursor + (isSelected ? theme.bold(truncatedMsg) : truncatedMsg);
            lines.push(messageLine);
            // Second line: metadata (position in history)
            const position = i + 1;
            const metadata = `  Message ${position} of ${this.messages.length}`;
            const metadataLine = theme.fg("muted", metadata);
            lines.push(metadataLine);
            lines.push(""); // Blank line between messages
        }
        // Add scroll indicator if needed
        if (startIndex > 0 || endIndex < this.messages.length) {
            const scrollInfo = theme.fg("muted", `  (${this.selectedIndex + 1}/${this.messages.length})`);
            lines.push(scrollInfo);
        }
        return lines;
    }
    handleInput(keyData) {
        const kb = getEditorKeybindings();
        // Up arrow - go to previous (older) message, wrap to bottom when at top
        if (kb.matches(keyData, "selectUp")) {
            this.selectedIndex = this.selectedIndex === 0 ? this.messages.length - 1 : this.selectedIndex - 1;
        }
        // Down arrow - go to next (newer) message, wrap to top when at bottom
        else if (kb.matches(keyData, "selectDown")) {
            this.selectedIndex = this.selectedIndex === this.messages.length - 1 ? 0 : this.selectedIndex + 1;
        }
        // Enter - select message and branch
        else if (kb.matches(keyData, "selectConfirm")) {
            const selected = this.messages[this.selectedIndex];
            if (selected && this.onSelect) {
                this.onSelect(selected.id);
            }
        }
        // Escape - cancel
        else if (kb.matches(keyData, "selectCancel")) {
            if (this.onCancel) {
                this.onCancel();
            }
        }
    }
}
/**
 * Component that renders a user message selector for branching
 */
export class UserMessageSelectorComponent extends Container {
    messageList;
    constructor(messages, onSelect, onCancel) {
        super();
        // Add header
        this.addChild(new Spacer(1));
        this.addChild(new Text(theme.bold("Branch from Message"), 1, 0));
        this.addChild(new Text(theme.fg("muted", "Select a message to create a new branch from that point"), 1, 0));
        this.addChild(new Spacer(1));
        this.addChild(new DynamicBorder());
        this.addChild(new Spacer(1));
        // Create message list
        this.messageList = new UserMessageList(messages);
        this.messageList.onSelect = onSelect;
        this.messageList.onCancel = onCancel;
        this.addChild(this.messageList);
        // Add bottom border
        this.addChild(new Spacer(1));
        this.addChild(new DynamicBorder());
        // Auto-cancel if no messages
        if (messages.length === 0) {
            setTimeout(() => onCancel(), 100);
        }
    }
    getMessageList() {
        return this.messageList;
    }
}
//# sourceMappingURL=user-message-selector.js.map