import Clipboard from "@mariozechner/clipboard";
import { spawnSync } from "child_process";
const PREFERRED_IMAGE_MIME_TYPES = ["image/png", "image/jpeg", "image/webp", "image/gif"];
const DEFAULT_LIST_TIMEOUT_MS = 1000;
const DEFAULT_READ_TIMEOUT_MS = 3000;
const DEFAULT_MAX_BUFFER_BYTES = 50 * 1024 * 1024;
export function isWaylandSession(env = process.env) {
    return Boolean(env.WAYLAND_DISPLAY) || env.XDG_SESSION_TYPE === "wayland";
}
function baseMimeType(mimeType) {
    return mimeType.split(";")[0]?.trim().toLowerCase() ?? mimeType.toLowerCase();
}
export function extensionForImageMimeType(mimeType) {
    switch (baseMimeType(mimeType)) {
        case "image/png":
            return "png";
        case "image/jpeg":
            return "jpg";
        case "image/webp":
            return "webp";
        case "image/gif":
            return "gif";
        default:
            return null;
    }
}
function selectPreferredImageMimeType(mimeTypes) {
    const normalized = mimeTypes
        .map((t) => t.trim())
        .filter(Boolean)
        .map((t) => ({ raw: t, base: baseMimeType(t) }));
    for (const preferred of PREFERRED_IMAGE_MIME_TYPES) {
        const match = normalized.find((t) => t.base === preferred);
        if (match) {
            return match.raw;
        }
    }
    const anyImage = normalized.find((t) => t.base.startsWith("image/"));
    return anyImage?.raw ?? null;
}
function runCommand(command, args, options) {
    const timeoutMs = options?.timeoutMs ?? DEFAULT_READ_TIMEOUT_MS;
    const maxBufferBytes = options?.maxBufferBytes ?? DEFAULT_MAX_BUFFER_BYTES;
    const result = spawnSync(command, args, {
        timeout: timeoutMs,
        maxBuffer: maxBufferBytes,
    });
    if (result.error) {
        return { ok: false, stdout: Buffer.alloc(0) };
    }
    if (result.status !== 0) {
        return { ok: false, stdout: Buffer.alloc(0) };
    }
    const stdout = Buffer.isBuffer(result.stdout)
        ? result.stdout
        : Buffer.from(result.stdout ?? "", typeof result.stdout === "string" ? "utf-8" : undefined);
    return { ok: true, stdout };
}
function readClipboardImageViaWlPaste() {
    const list = runCommand("wl-paste", ["--list-types"], { timeoutMs: DEFAULT_LIST_TIMEOUT_MS });
    if (!list.ok) {
        return null;
    }
    const types = list.stdout
        .toString("utf-8")
        .split(/\r?\n/)
        .map((t) => t.trim())
        .filter(Boolean);
    const selectedType = selectPreferredImageMimeType(types);
    if (!selectedType) {
        return null;
    }
    const data = runCommand("wl-paste", ["--type", selectedType, "--no-newline"]);
    if (!data.ok || data.stdout.length === 0) {
        return null;
    }
    return { bytes: data.stdout, mimeType: baseMimeType(selectedType) };
}
function readClipboardImageViaXclip() {
    const targets = runCommand("xclip", ["-selection", "clipboard", "-t", "TARGETS", "-o"], {
        timeoutMs: DEFAULT_LIST_TIMEOUT_MS,
    });
    let candidateTypes = [];
    if (targets.ok) {
        candidateTypes = targets.stdout
            .toString("utf-8")
            .split(/\r?\n/)
            .map((t) => t.trim())
            .filter(Boolean);
    }
    const preferred = candidateTypes.length > 0 ? selectPreferredImageMimeType(candidateTypes) : null;
    const tryTypes = preferred ? [preferred, ...PREFERRED_IMAGE_MIME_TYPES] : [...PREFERRED_IMAGE_MIME_TYPES];
    for (const mimeType of tryTypes) {
        const data = runCommand("xclip", ["-selection", "clipboard", "-t", mimeType, "-o"]);
        if (data.ok && data.stdout.length > 0) {
            return { bytes: data.stdout, mimeType: baseMimeType(mimeType) };
        }
    }
    return null;
}
export async function readClipboardImage(options) {
    const env = options?.env ?? process.env;
    const platform = options?.platform ?? process.platform;
    if (platform === "linux" && isWaylandSession(env)) {
        return readClipboardImageViaWlPaste() ?? readClipboardImageViaXclip();
    }
    if (!Clipboard.hasImage()) {
        return null;
    }
    const imageData = await Clipboard.getImageBinary();
    if (!imageData || imageData.length === 0) {
        return null;
    }
    const bytes = imageData instanceof Uint8Array ? imageData : Uint8Array.from(imageData);
    return { bytes, mimeType: "image/png" };
}
//# sourceMappingURL=clipboard-image.js.map