import { $ark, writeUnboundableMessage } from "@ark/schema";
import { isKeyOf, throwParseError } from "@ark/util";
import { invertedComparators, maxComparators, writeUnpairableComparatorMessage } from "../../reduce/shared.js";
import { extractDateLiteralSource, isDateLiteral } from "../operand/date.js";
export const parseBound = (s, start) => {
    const comparator = shiftComparator(s, start);
    if (s.root.hasKind("unit")) {
        if (typeof s.root.unit === "number") {
            s.reduceLeftBound(s.root.unit, comparator);
            s.unsetRoot();
            return;
        }
        if (s.root.unit instanceof Date) {
            const literal = `d'${s.root.description ?? s.root.unit.toISOString()}'`;
            s.unsetRoot();
            s.reduceLeftBound(literal, comparator);
            return;
        }
    }
    return parseRightBound(s, comparator);
};
export const comparatorStartChars = {
    "<": 1,
    ">": 1,
    "=": 1
};
const shiftComparator = (s, start) => s.scanner.lookaheadIs("=") ?
    `${start}${s.scanner.shift()}`
    : start;
export const writeIncompatibleRangeMessage = (l, r) => `Bound kinds ${l} and ${r} are incompatible`;
export const getBoundKinds = (comparator, limit, root, boundKind) => {
    if (root.extends($ark.intrinsic.number)) {
        if (typeof limit !== "number") {
            return throwParseError(writeInvalidLimitMessage(comparator, limit, boundKind));
        }
        return (comparator === "==" ? ["min", "max"]
            : comparator[0] === ">" ? ["min"]
                : ["max"]);
    }
    if (root.extends($ark.intrinsic.lengthBoundable)) {
        if (typeof limit !== "number") {
            return throwParseError(writeInvalidLimitMessage(comparator, limit, boundKind));
        }
        return (comparator === "==" ? ["exactLength"]
            : comparator[0] === ">" ? ["minLength"]
                : ["maxLength"]);
    }
    if (root.extends($ark.intrinsic.Date)) {
        // allow either numeric or date limits
        return (comparator === "==" ? ["after", "before"]
            : comparator[0] === ">" ? ["after"]
                : ["before"]);
    }
    return throwParseError(writeUnboundableMessage(root.expression));
};
const openLeftBoundToRoot = (leftBound) => ({
    rule: isDateLiteral(leftBound.limit) ?
        extractDateLiteralSource(leftBound.limit)
        : leftBound.limit,
    exclusive: leftBound.comparator.length === 1
});
export const parseRightBound = (s, comparator) => {
    // store the node that will be bounded
    const previousRoot = s.unsetRoot();
    const previousScannerIndex = s.scanner.location;
    s.parseOperand();
    const limitNode = s.unsetRoot();
    // after parsing the next operand, use the locations to get the
    // token from which it was parsed
    const limitToken = s.scanner.sliceChars(previousScannerIndex, s.scanner.location);
    s.root = previousRoot;
    if (!limitNode.hasKind("unit") ||
        (typeof limitNode.unit !== "number" && !(limitNode.unit instanceof Date)))
        return s.error(writeInvalidLimitMessage(comparator, limitToken, "right"));
    const limit = limitNode.unit;
    // apply the newly-parsed right bound
    const exclusive = comparator.length === 1;
    const boundKinds = getBoundKinds(comparator, typeof limit === "number" ? limit : limitToken, previousRoot, "right");
    for (const kind of boundKinds) {
        s.constrainRoot(kind, comparator === "==" ? { rule: limit } : { rule: limit, exclusive });
    }
    if (!s.branches.leftBound)
        return;
    // if there's an open left bound, perform additional validation and apply it
    if (!isKeyOf(comparator, maxComparators))
        return s.error(writeUnpairableComparatorMessage(comparator));
    const lowerBoundKind = getBoundKinds(s.branches.leftBound.comparator, s.branches.leftBound.limit, previousRoot, "left");
    s.constrainRoot(lowerBoundKind[0], openLeftBoundToRoot(s.branches.leftBound));
    s.branches.leftBound = null;
};
export const writeInvalidLimitMessage = (comparator, limit, boundKind) => `Comparator ${boundKind === "left" ? invertedComparators[comparator] : comparator} must be ${boundKind === "left" ? "preceded" : "followed"} by a corresponding literal (was ${limit})`;
