import { $ark, BaseScope, hasArkKind, parseGeneric } from "@ark/schema";
import { Scanner, enumValues, flatMorph, isArray, isThunk, throwParseError } from "@ark/util";
import { InternalFnParser } from "./fn.js";
import { parseGenericParamName } from "./generic.js";
import { InternalMatchParser } from "./match.js";
import { shallowDefaultableMessage, shallowOptionalMessage } from "./parser/ast/validate.js";
import { parseInnerDefinition } from "./parser/definition.js";
import { InternalTypeParser } from "./type.js";
export const $arkTypeRegistry = $ark;
export class InternalScope extends BaseScope {
    get ambientAttachments() {
        if (!$arkTypeRegistry.typeAttachments)
            return;
        return this.cacheGetter("ambientAttachments", flatMorph($arkTypeRegistry.typeAttachments, (k, v) => [
            k,
            this.bindReference(v)
        ]));
    }
    preparseOwnAliasEntry(alias, def) {
        const firstParamIndex = alias.indexOf("<");
        if (firstParamIndex === -1) {
            if (hasArkKind(def, "module") || hasArkKind(def, "generic"))
                return [alias, def];
            const qualifiedName = this.name === "ark" ? alias
                : alias === "root" ? this.name
                    : `${this.name}.${alias}`;
            const config = this.resolvedConfig.keywords?.[qualifiedName];
            if (config)
                def = [def, "@", config];
            return [alias, def];
        }
        if (alias[alias.length - 1] !== ">") {
            throwParseError(`'>' must be the last character of a generic declaration in a scope`);
        }
        const name = alias.slice(0, firstParamIndex);
        const paramString = alias.slice(firstParamIndex + 1, -1);
        return [
            name,
            // use a thunk definition for the generic so that we can parse
            // constraints within the current scope
            () => {
                const params = this.parseGenericParams(paramString, { alias: name });
                const generic = parseGeneric(params, def, this);
                return generic;
            }
        ];
    }
    parseGenericParams(def, opts) {
        return parseGenericParamName(new Scanner(def), [], this.createParseContext({
            ...opts,
            def,
            prefix: "generic"
        }));
    }
    normalizeRootScopeValue(resolution) {
        if (isThunk(resolution) && !hasArkKind(resolution, "generic"))
            return resolution();
        return resolution;
    }
    preparseOwnDefinitionFormat(def, opts) {
        return {
            ...opts,
            def,
            prefix: opts.alias ?? "type"
        };
    }
    parseOwnDefinitionFormat(def, ctx) {
        const isScopeAlias = ctx.alias && ctx.alias in this.aliases;
        // if the definition being parsed is not a scope alias and is not a
        // generic instantiation (i.e. opts don't include args), add `this` as a resolution.
        // if we're parsing a nested string, ctx.args will have already been set
        if (!isScopeAlias && !ctx.args)
            ctx.args = { this: ctx.id };
        const result = parseInnerDefinition(def, ctx);
        if (isArray(result)) {
            if (result[1] === "=")
                return throwParseError(shallowDefaultableMessage);
            if (result[1] === "?")
                return throwParseError(shallowOptionalMessage);
        }
        return result;
    }
    unit = value => this.units([value]);
    valueOf = tsEnum => this.units(enumValues(tsEnum));
    enumerated = (...values) => this.units(values);
    instanceOf = ctor => this.node("proto", { proto: ctor }, { prereduced: true });
    or = (...defs) => this.schema(defs.map(def => this.parse(def)));
    and = (...defs) => defs.reduce((node, def) => node.and(this.parse(def)), this.intrinsic.unknown);
    merge = (...defs) => defs.reduce((node, def) => node.merge(this.parse(def)), this.intrinsic.object);
    pipe = (...morphs) => this.intrinsic.unknown.pipe(...morphs);
    fn = new InternalFnParser(this);
    match = new InternalMatchParser(this);
    declare = () => ({
        type: this.type
    });
    define(def) {
        return def;
    }
    type = new InternalTypeParser(this);
    static scope = ((def, config = {}) => new InternalScope(def, config));
    static module = ((def, config = {}) => this.scope(def, config).export());
}
export const scope = Object.assign(InternalScope.scope, {
    define: (def) => def
});
export const Scope = InternalScope;
