import HTTPClient from "./http-axios.js";
import * as Types from "./types.js";
import { createMultipartFormData, ensureJSON, toArray } from "./utils.js";
import { DATA_API_PREFIX, MESSAGING_API_PREFIX, OAUTH_BASE_PREFIX, OAUTH_BASE_PREFIX_V2_1, } from "./endpoints.js";
/**
 * @deprecated Use clients generated by openapi spec instead.
 */
export default class Client {
    config;
    http;
    requestOption = {};
    constructor(config) {
        if (!config.channelAccessToken) {
            throw new Error("no channel access token");
        }
        this.config = config;
        this.http = new HTTPClient({
            defaultHeaders: {
                Authorization: "Bearer " + this.config.channelAccessToken,
            },
            responseParser: this.parseHTTPResponse.bind(this),
            ...config.httpConfig,
        });
    }
    setRequestOptionOnce(option) {
        this.requestOption = option;
    }
    generateRequestConfig() {
        const config = { headers: {} };
        if (this.requestOption.retryKey) {
            config.headers["X-Line-Retry-Key"] = this.requestOption.retryKey;
        }
        // clear requestOption
        this.requestOption = {};
        return config;
    }
    parseHTTPResponse(response) {
        const { LINE_REQUEST_ID_HTTP_HEADER_NAME } = Types;
        let resBody = {
            ...response.data,
        };
        if (response.headers[LINE_REQUEST_ID_HTTP_HEADER_NAME]) {
            resBody[LINE_REQUEST_ID_HTTP_HEADER_NAME] =
                response.headers[LINE_REQUEST_ID_HTTP_HEADER_NAME];
        }
        return resBody;
    }
    pushMessage(to, messages, notificationDisabled = false, customAggregationUnits) {
        return this.http.post(`${MESSAGING_API_PREFIX}/message/push`, {
            messages: toArray(messages),
            to,
            notificationDisabled,
            customAggregationUnits,
        }, this.generateRequestConfig());
    }
    replyMessage(replyToken, messages, notificationDisabled = false) {
        return this.http.post(`${MESSAGING_API_PREFIX}/message/reply`, {
            messages: toArray(messages),
            replyToken,
            notificationDisabled,
        });
    }
    async multicast(to, messages, notificationDisabled = false, customAggregationUnits) {
        return this.http.post(`${MESSAGING_API_PREFIX}/message/multicast`, {
            messages: toArray(messages),
            to,
            notificationDisabled,
            customAggregationUnits,
        }, this.generateRequestConfig());
    }
    async narrowcast(messages, recipient, filter, limit, notificationDisabled) {
        return this.http.post(`${MESSAGING_API_PREFIX}/message/narrowcast`, {
            messages: toArray(messages),
            recipient,
            filter,
            limit,
            notificationDisabled,
        }, this.generateRequestConfig());
    }
    async broadcast(messages, notificationDisabled = false) {
        return this.http.post(`${MESSAGING_API_PREFIX}/message/broadcast`, {
            messages: toArray(messages),
            notificationDisabled,
        }, this.generateRequestConfig());
    }
    validatePushMessageObjects(messages) {
        return this.http.post(`${MESSAGING_API_PREFIX}/message/validate/push`, {
            messages: toArray(messages),
        }, this.generateRequestConfig());
    }
    validateReplyMessageObjects(messages) {
        return this.http.post(`${MESSAGING_API_PREFIX}/message/validate/reply`, {
            messages: toArray(messages),
        });
    }
    async validateMulticastMessageObjects(messages) {
        return this.http.post(`${MESSAGING_API_PREFIX}/message/validate/multicast`, {
            messages: toArray(messages),
        }, this.generateRequestConfig());
    }
    async validateNarrowcastMessageObjects(messages) {
        return this.http.post(`${MESSAGING_API_PREFIX}/message/validate/narrowcast`, {
            messages: toArray(messages),
        }, this.generateRequestConfig());
    }
    async validateBroadcastMessageObjects(messages) {
        return this.http.post(`${MESSAGING_API_PREFIX}/message/validate/broadcast`, {
            messages: toArray(messages),
        }, this.generateRequestConfig());
    }
    validateCustomAggregationUnits(units) {
        const messages = [];
        if (units.length > 1) {
            messages.push("customAggregationUnits can only contain one unit");
        }
        units.forEach((unit, index) => {
            if (unit.length > 30) {
                messages.push(`customAggregationUnits[${index}] must be less than or equal to 30 characters`);
            }
            if (!/^[a-zA-Z0-9_]+$/.test(unit)) {
                messages.push(`customAggregationUnits[${index}] must be alphanumeric characters or underscores`);
            }
        });
        return {
            messages,
            valid: messages.length === 0,
        };
    }
    async getProfile(userId) {
        const profile = await this.http.get(`${MESSAGING_API_PREFIX}/profile/${userId}`);
        return ensureJSON(profile);
    }
    async getChatMemberProfile(chatType, chatId, userId) {
        const profile = await this.http.get(`${MESSAGING_API_PREFIX}/${chatType}/${chatId}/member/${userId}`);
        return ensureJSON(profile);
    }
    async getGroupMemberProfile(groupId, userId) {
        return this.getChatMemberProfile("group", groupId, userId);
    }
    async getRoomMemberProfile(roomId, userId) {
        return this.getChatMemberProfile("room", roomId, userId);
    }
    async getChatMemberIds(chatType, chatId) {
        let memberIds = [];
        let start;
        do {
            const res = await this.http.get(`${MESSAGING_API_PREFIX}/${chatType}/${chatId}/members/ids`, start ? { start } : null);
            ensureJSON(res);
            memberIds = memberIds.concat(res.memberIds);
            start = res.next;
        } while (start);
        return memberIds;
    }
    async getGroupMemberIds(groupId) {
        return this.getChatMemberIds("group", groupId);
    }
    async getRoomMemberIds(roomId) {
        return this.getChatMemberIds("room", roomId);
    }
    async getBotFollowersIds() {
        let userIds = [];
        let start;
        do {
            const res = await this.http.get(`${MESSAGING_API_PREFIX}/followers/ids`, start ? { start, limit: 1000 } : { limit: 1000 });
            ensureJSON(res);
            userIds = userIds.concat(res.userIds);
            start = res.next;
        } while (start);
        return userIds;
    }
    async getGroupMembersCount(groupId) {
        const groupMemberCount = await this.http.get(`${MESSAGING_API_PREFIX}/group/${groupId}/members/count`);
        return ensureJSON(groupMemberCount);
    }
    async getRoomMembersCount(roomId) {
        const roomMemberCount = await this.http.get(`${MESSAGING_API_PREFIX}/room/${roomId}/members/count`);
        return ensureJSON(roomMemberCount);
    }
    async getGroupSummary(groupId) {
        const groupSummary = await this.http.get(`${MESSAGING_API_PREFIX}/group/${groupId}/summary`);
        return ensureJSON(groupSummary);
    }
    async getMessageContent(messageId) {
        return this.http.getStream(`${DATA_API_PREFIX}/message/${messageId}/content`);
    }
    leaveChat(chatType, chatId) {
        return this.http.post(`${MESSAGING_API_PREFIX}/${chatType}/${chatId}/leave`);
    }
    async leaveGroup(groupId) {
        return this.leaveChat("group", groupId);
    }
    async leaveRoom(roomId) {
        return this.leaveChat("room", roomId);
    }
    async getRichMenu(richMenuId) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/richmenu/${richMenuId}`);
        return ensureJSON(res);
    }
    async createRichMenu(richMenu) {
        const res = await this.http.post(`${MESSAGING_API_PREFIX}/richmenu`, richMenu);
        return ensureJSON(res).richMenuId;
    }
    async deleteRichMenu(richMenuId) {
        return this.http.delete(`${MESSAGING_API_PREFIX}/richmenu/${richMenuId}`);
    }
    async getRichMenuAliasList() {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/richmenu/alias/list`);
        return ensureJSON(res);
    }
    async getRichMenuAlias(richMenuAliasId) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/richmenu/alias/${richMenuAliasId}`);
        return ensureJSON(res);
    }
    async createRichMenuAlias(richMenuId, richMenuAliasId) {
        const res = await this.http.post(`${MESSAGING_API_PREFIX}/richmenu/alias`, {
            richMenuId,
            richMenuAliasId,
        });
        return ensureJSON(res);
    }
    async deleteRichMenuAlias(richMenuAliasId) {
        const res = this.http.delete(`${MESSAGING_API_PREFIX}/richmenu/alias/${richMenuAliasId}`);
        return ensureJSON(res);
    }
    async updateRichMenuAlias(richMenuAliasId, richMenuId) {
        const res = await this.http.post(`${MESSAGING_API_PREFIX}/richmenu/alias/${richMenuAliasId}`, {
            richMenuId,
        });
        return ensureJSON(res);
    }
    async getRichMenuIdOfUser(userId) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/user/${userId}/richmenu`);
        return ensureJSON(res).richMenuId;
    }
    async linkRichMenuToUser(userId, richMenuId) {
        return this.http.post(`${MESSAGING_API_PREFIX}/user/${userId}/richmenu/${richMenuId}`);
    }
    async unlinkRichMenuFromUser(userId) {
        return this.http.delete(`${MESSAGING_API_PREFIX}/user/${userId}/richmenu`);
    }
    async linkRichMenuToMultipleUsers(richMenuId, userIds) {
        return this.http.post(`${MESSAGING_API_PREFIX}/richmenu/bulk/link`, {
            richMenuId,
            userIds,
        });
    }
    async unlinkRichMenusFromMultipleUsers(userIds) {
        return this.http.post(`${MESSAGING_API_PREFIX}/richmenu/bulk/unlink`, {
            userIds,
        });
    }
    async getRichMenuImage(richMenuId) {
        return this.http.getStream(`${DATA_API_PREFIX}/richmenu/${richMenuId}/content`);
    }
    async setRichMenuImage(richMenuId, data, contentType) {
        return this.http.postBinary(`${DATA_API_PREFIX}/richmenu/${richMenuId}/content`, data, contentType);
    }
    async getRichMenuList() {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/richmenu/list`);
        return ensureJSON(res).richmenus;
    }
    async setDefaultRichMenu(richMenuId) {
        return this.http.post(`${MESSAGING_API_PREFIX}/user/all/richmenu/${richMenuId}`);
    }
    async getDefaultRichMenuId() {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/user/all/richmenu`);
        return ensureJSON(res).richMenuId;
    }
    async deleteDefaultRichMenu() {
        return this.http.delete(`${MESSAGING_API_PREFIX}/user/all/richmenu`);
    }
    async getLinkToken(userId) {
        const res = await this.http.post(`${MESSAGING_API_PREFIX}/user/${userId}/linkToken`);
        return ensureJSON(res).linkToken;
    }
    async getNumberOfSentReplyMessages(date) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/message/delivery/reply?date=${date}`);
        return ensureJSON(res);
    }
    async getNumberOfSentPushMessages(date) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/message/delivery/push?date=${date}`);
        return ensureJSON(res);
    }
    async getNumberOfSentMulticastMessages(date) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/message/delivery/multicast?date=${date}`);
        return ensureJSON(res);
    }
    async getNarrowcastProgress(requestId) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/message/progress/narrowcast?requestId=${requestId}`);
        return ensureJSON(res);
    }
    async getTargetLimitForAdditionalMessages() {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/message/quota`);
        return ensureJSON(res);
    }
    async getNumberOfMessagesSentThisMonth() {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/message/quota/consumption`);
        return ensureJSON(res);
    }
    async getNumberOfSentBroadcastMessages(date) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/message/delivery/broadcast?date=${date}`);
        return ensureJSON(res);
    }
    async getNumberOfMessageDeliveries(date) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/insight/message/delivery?date=${date}`);
        return ensureJSON(res);
    }
    async getNumberOfFollowers(date) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/insight/followers?date=${date}`);
        return ensureJSON(res);
    }
    async getFriendDemographics() {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/insight/demographic`);
        return ensureJSON(res);
    }
    async getUserInteractionStatistics(requestId) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/insight/message/event?requestId=${requestId}`);
        return ensureJSON(res);
    }
    async getStatisticsPerUnit(customAggregationUnit, from, to) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/insight/message/event/aggregation?customAggregationUnit=${customAggregationUnit}&from=${from}&to=${to}`);
        return ensureJSON(res);
    }
    async createUploadAudienceGroup(uploadAudienceGroup) {
        const res = await this.http.post(`${MESSAGING_API_PREFIX}/audienceGroup/upload`, {
            ...uploadAudienceGroup,
        });
        return ensureJSON(res);
    }
    async createUploadAudienceGroupByFile(uploadAudienceGroup) {
        const file = await this.http.toBuffer(uploadAudienceGroup.file);
        const body = createMultipartFormData({ ...uploadAudienceGroup, file });
        const res = await this.http.postFormMultipart(`${DATA_API_PREFIX}/audienceGroup/upload/byFile`, body);
        return ensureJSON(res);
    }
    async updateUploadAudienceGroup(uploadAudienceGroup, 
    // for set request timeout
    httpConfig) {
        const res = await this.http.put(`${MESSAGING_API_PREFIX}/audienceGroup/upload`, {
            ...uploadAudienceGroup,
        }, httpConfig);
        return ensureJSON(res);
    }
    async updateUploadAudienceGroupByFile(uploadAudienceGroup, 
    // for set request timeout
    httpConfig) {
        const file = await this.http.toBuffer(uploadAudienceGroup.file);
        const body = createMultipartFormData({ ...uploadAudienceGroup, file });
        const res = await this.http.putFormMultipart(`${DATA_API_PREFIX}/audienceGroup/upload/byFile`, body, httpConfig);
        return ensureJSON(res);
    }
    async createClickAudienceGroup(clickAudienceGroup) {
        const res = await this.http.post(`${MESSAGING_API_PREFIX}/audienceGroup/click`, {
            ...clickAudienceGroup,
        });
        return ensureJSON(res);
    }
    async createImpAudienceGroup(impAudienceGroup) {
        const res = await this.http.post(`${MESSAGING_API_PREFIX}/audienceGroup/imp`, {
            ...impAudienceGroup,
        });
        return ensureJSON(res);
    }
    async setDescriptionAudienceGroup(description, audienceGroupId) {
        const res = await this.http.put(`${MESSAGING_API_PREFIX}/audienceGroup/${audienceGroupId}/updateDescription`, {
            description,
        });
        return ensureJSON(res);
    }
    async deleteAudienceGroup(audienceGroupId) {
        const res = await this.http.delete(`${MESSAGING_API_PREFIX}/audienceGroup/${audienceGroupId}`);
        return ensureJSON(res);
    }
    async getAudienceGroup(audienceGroupId) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/audienceGroup/${audienceGroupId}`);
        return ensureJSON(res);
    }
    async getAudienceGroups(page, description, status, size, createRoute, includesExternalPublicGroups) {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/audienceGroup/list`, {
            page,
            description,
            status,
            size,
            createRoute,
            includesExternalPublicGroups,
        });
        return ensureJSON(res);
    }
    async getAudienceGroupAuthorityLevel() {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/audienceGroup/authorityLevel`);
        return ensureJSON(res);
    }
    async changeAudienceGroupAuthorityLevel(authorityLevel) {
        const res = await this.http.put(`${MESSAGING_API_PREFIX}/audienceGroup/authorityLevel`, { authorityLevel });
        return ensureJSON(res);
    }
    async getBotInfo() {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/info`);
        return ensureJSON(res);
    }
    async setWebhookEndpointUrl(endpoint) {
        return this.http.put(`${MESSAGING_API_PREFIX}/channel/webhook/endpoint`, { endpoint });
    }
    async getWebhookEndpointInfo() {
        const res = await this.http.get(`${MESSAGING_API_PREFIX}/channel/webhook/endpoint`);
        return ensureJSON(res);
    }
    async testWebhookEndpoint(endpoint) {
        const res = await this.http.post(`${MESSAGING_API_PREFIX}/channel/webhook/test`, { endpoint });
        return ensureJSON(res);
    }
    async validateRichMenu(richMenu) {
        return await this.http.post(`${MESSAGING_API_PREFIX}/richmenu/validate`, richMenu);
    }
}
export class OAuth {
    http;
    constructor() {
        this.http = new HTTPClient();
    }
    issueAccessToken(client_id, client_secret) {
        return this.http.postForm(`${OAUTH_BASE_PREFIX}/accessToken`, {
            grant_type: "client_credentials",
            client_id,
            client_secret,
        });
    }
    revokeAccessToken(access_token) {
        return this.http.postForm(`${OAUTH_BASE_PREFIX}/revoke`, { access_token });
    }
    verifyAccessToken(access_token) {
        return this.http.get(`${OAUTH_BASE_PREFIX_V2_1}/verify`, { access_token });
    }
    verifyIdToken(id_token, client_id, nonce, user_id) {
        const body = {
            id_token,
            client_id,
        };
        if (nonce) {
            body.nonce = nonce;
        }
        if (user_id) {
            body.user_id = user_id;
        }
        return this.http.postForm(`${OAUTH_BASE_PREFIX_V2_1}/verify`, body);
    }
    issueChannelAccessTokenV2_1(client_assertion) {
        return this.http.postForm(`${OAUTH_BASE_PREFIX_V2_1}/token`, {
            grant_type: "client_credentials",
            client_assertion_type: "urn:ietf:params:oauth:client-assertion-type:jwt-bearer",
            client_assertion,
        });
    }
    getChannelAccessTokenKeyIdsV2_1(client_assertion) {
        return this.http.get(`${OAUTH_BASE_PREFIX_V2_1}/tokens/kid`, {
            client_assertion_type: "urn:ietf:params:oauth:client-assertion-type:jwt-bearer",
            client_assertion,
        });
    }
    revokeChannelAccessTokenV2_1(client_id, client_secret, access_token) {
        return this.http.postForm(`${OAUTH_BASE_PREFIX_V2_1}/revoke`, {
            client_id,
            client_secret,
            access_token,
        });
    }
}
//# sourceMappingURL=client.js.map