import * as schema from "./schema/index.js";
export type * from "./schema/types.gen.js";
export * from "./schema/index.js";
export * from "./stream.js";
import type { Stream } from "./stream.js";
import type { Result, ErrorResponse, RequestHandler, NotificationHandler } from "./jsonrpc.js";
/**
 * An agent-side connection to a client.
 *
 * This class provides the agent's view of an ACP connection, allowing
 * agents to communicate with clients. It implements the {@link Client} interface
 * to provide methods for requesting permissions, accessing the file system,
 * and sending session updates.
 *
 * See protocol docs: [Agent](https://agentclientprotocol.com/protocol/overview#agent)
 */
export declare class AgentSideConnection {
    #private;
    /**
     * Creates a new agent-side connection to a client.
     *
     * This establishes the communication channel from the agent's perspective
     * following the ACP specification.
     *
     * @param toAgent - A function that creates an Agent handler to process incoming client requests
     * @param stream - The bidirectional message stream for communication. Typically created using
     *                 {@link ndJsonStream} for stdio-based connections.
     *
     * See protocol docs: [Communication Model](https://agentclientprotocol.com/protocol/overview#communication-model)
     */
    constructor(toAgent: (conn: AgentSideConnection) => Agent, stream: Stream);
    /**
     * Handles session update notifications from the agent.
     *
     * This is a notification endpoint (no response expected) that sends
     * real-time updates about session progress, including message chunks,
     * tool calls, and execution plans.
     *
     * Note: Clients SHOULD continue accepting tool call updates even after
     * sending a `session/cancel` notification, as the agent may send final
     * updates before responding with the cancelled stop reason.
     *
     * See protocol docs: [Agent Reports Output](https://agentclientprotocol.com/protocol/prompt-turn#3-agent-reports-output)
     */
    sessionUpdate(params: schema.SessionNotification): Promise<void>;
    /**
     * Requests permission from the user for a tool call operation.
     *
     * Called by the agent when it needs user authorization before executing
     * a potentially sensitive operation. The client should present the options
     * to the user and return their decision.
     *
     * If the client cancels the prompt turn via `session/cancel`, it MUST
     * respond to this request with `RequestPermissionOutcome::Cancelled`.
     *
     * See protocol docs: [Requesting Permission](https://agentclientprotocol.com/protocol/tool-calls#requesting-permission)
     */
    requestPermission(params: schema.RequestPermissionRequest): Promise<schema.RequestPermissionResponse>;
    /**
     * Reads content from a text file in the client's file system.
     *
     * Only available if the client advertises the `fs.readTextFile` capability.
     * Allows the agent to access file contents within the client's environment.
     *
     * See protocol docs: [Client](https://agentclientprotocol.com/protocol/overview#client)
     */
    readTextFile(params: schema.ReadTextFileRequest): Promise<schema.ReadTextFileResponse>;
    /**
     * Writes content to a text file in the client's file system.
     *
     * Only available if the client advertises the `fs.writeTextFile` capability.
     * Allows the agent to create or modify files within the client's environment.
     *
     * See protocol docs: [Client](https://agentclientprotocol.com/protocol/overview#client)
     */
    writeTextFile(params: schema.WriteTextFileRequest): Promise<schema.WriteTextFileResponse>;
    /**
     * Executes a command in a new terminal.
     *
     * Returns a `TerminalHandle` that can be used to get output, wait for exit,
     * kill the command, or release the terminal.
     *
     * The terminal can also be embedded in tool calls by using its ID in
     * `ToolCallContent` with type "terminal".
     *
     * @param params - The terminal creation parameters
     * @returns A handle to control and monitor the terminal
     */
    createTerminal(params: schema.CreateTerminalRequest): Promise<TerminalHandle>;
    /**
     * Extension method
     *
     * Allows the Agent to send an arbitrary request that is not part of the ACP spec.
     */
    extMethod(method: string, params: Record<string, unknown>): Promise<Record<string, unknown>>;
    /**
     * Extension notification
     *
     * Allows the Agent to send an arbitrary notification that is not part of the ACP spec.
     */
    extNotification(method: string, params: Record<string, unknown>): Promise<void>;
    /**
     * AbortSignal that aborts when the connection closes.
     *
     * This signal can be used to:
     * - Listen for connection closure: `connection.signal.addEventListener('abort', () => {...})`
     * - Check connection status synchronously: `if (connection.signal.aborted) {...}`
     * - Pass to other APIs (fetch, setTimeout) for automatic cancellation
     *
     * The connection closes when the underlying stream ends, either normally or due to an error.
     *
     * @example
     * ```typescript
     * const connection = new AgentSideConnection(agent, stream);
     *
     * // Listen for closure
     * connection.signal.addEventListener('abort', () => {
     *   console.log('Connection closed - performing cleanup');
     * });
     *
     * // Check status
     * if (connection.signal.aborted) {
     *   console.log('Connection is already closed');
     * }
     *
     * // Pass to other APIs
     * fetch(url, { signal: connection.signal });
     * ```
     */
    get signal(): AbortSignal;
    /**
     * Promise that resolves when the connection closes.
     *
     * The connection closes when the underlying stream ends, either normally or due to an error.
     * Once closed, the connection cannot send or receive any more messages.
     *
     * This is useful for async/await style cleanup:
     *
     * @example
     * ```typescript
     * const connection = new AgentSideConnection(agent, stream);
     * await connection.closed;
     * console.log('Connection closed - performing cleanup');
     * ```
     */
    get closed(): Promise<void>;
}
/**
 * Handle for controlling and monitoring a terminal created via `createTerminal`.
 *
 * Provides methods to:
 * - Get current output without waiting
 * - Wait for command completion
 * - Kill the running command
 * - Release terminal resources
 *
 * **Important:** Always call `release()` when done with the terminal to free resources.

 * The terminal supports async disposal via `Symbol.asyncDispose` for automatic cleanup.

 * You can use `await using` to ensure the terminal is automatically released when it
 * goes out of scope.
 */
export declare class TerminalHandle {
    #private;
    id: string;
    constructor(id: string, sessionId: string, conn: Connection);
    /**
     * Gets the current terminal output without waiting for the command to exit.
     */
    currentOutput(): Promise<schema.TerminalOutputResponse>;
    /**
     * Waits for the terminal command to complete and returns its exit status.
     */
    waitForExit(): Promise<schema.WaitForTerminalExitResponse>;
    /**
     * Kills the terminal command without releasing the terminal.
     *
     * The terminal remains valid after killing, allowing you to:
     * - Get the final output with `currentOutput()`
     * - Check the exit status
     * - Release the terminal when done
     *
     * Useful for implementing timeouts or cancellation.
     */
    kill(): Promise<schema.KillTerminalCommandResponse>;
    /**
     * Releases the terminal and frees all associated resources.
     *
     * If the command is still running, it will be killed.
     * After release, the terminal ID becomes invalid and cannot be used
     * with other terminal methods.
     *
     * Tool calls that already reference this terminal will continue to
     * display its output.
     *
     * **Important:** Always call this method when done with the terminal.
     */
    release(): Promise<schema.ReleaseTerminalResponse | void>;
    [Symbol.asyncDispose](): Promise<void>;
}
/**
 * A client-side connection to an agent.
 *
 * This class provides the client's view of an ACP connection, allowing
 * clients (such as code editors) to communicate with agents. It implements
 * the {@link Agent} interface to provide methods for initializing sessions, sending
 * prompts, and managing the agent lifecycle.
 *
 * See protocol docs: [Client](https://agentclientprotocol.com/protocol/overview#client)
 */
export declare class ClientSideConnection implements Agent {
    #private;
    /**
     * Creates a new client-side connection to an agent.
     *
     * This establishes the communication channel between a client and agent
     * following the ACP specification.
     *
     * @param toClient - A function that creates a Client handler to process incoming agent requests
     * @param stream - The bidirectional message stream for communication. Typically created using
     *                 {@link ndJsonStream} for stdio-based connections.
     *
     * See protocol docs: [Communication Model](https://agentclientprotocol.com/protocol/overview#communication-model)
     */
    constructor(toClient: (agent: Agent) => Client, stream: Stream);
    /**
     * Establishes the connection with a client and negotiates protocol capabilities.
     *
     * This method is called once at the beginning of the connection to:
     * - Negotiate the protocol version to use
     * - Exchange capability information between client and agent
     * - Determine available authentication methods
     *
     * The agent should respond with its supported protocol version and capabilities.
     *
     * See protocol docs: [Initialization](https://agentclientprotocol.com/protocol/initialization)
     */
    initialize(params: schema.InitializeRequest): Promise<schema.InitializeResponse>;
    /**
     * Creates a new conversation session with the agent.
     *
     * Sessions represent independent conversation contexts with their own history and state.
     *
     * The agent should:
     * - Create a new session context
     * - Connect to any specified MCP servers
     * - Return a unique session ID for future requests
     *
     * May return an `auth_required` error if the agent requires authentication.
     *
     * See protocol docs: [Session Setup](https://agentclientprotocol.com/protocol/session-setup)
     */
    newSession(params: schema.NewSessionRequest): Promise<schema.NewSessionResponse>;
    /**
     * Loads an existing session to resume a previous conversation.
     *
     * This method is only available if the agent advertises the `loadSession` capability.
     *
     * The agent should:
     * - Restore the session context and conversation history
     * - Connect to the specified MCP servers
     * - Stream the entire conversation history back to the client via notifications
     *
     * See protocol docs: [Loading Sessions](https://agentclientprotocol.com/protocol/session-setup#loading-sessions)
     */
    loadSession(params: schema.LoadSessionRequest): Promise<schema.LoadSessionResponse>;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Forks an existing session to create a new independent session.
     *
     * Creates a new session based on the context of an existing one, allowing
     * operations like generating summaries without affecting the original session's history.
     *
     * This method is only available if the agent advertises the `session.fork` capability.
     *
     * @experimental
     */
    unstable_forkSession(params: schema.ForkSessionRequest): Promise<schema.ForkSessionResponse>;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Lists existing sessions from the agent.
     *
     * This method is only available if the agent advertises the `listSessions` capability.
     *
     * Returns a list of sessions with metadata like session ID, working directory,
     * title, and last update time. Supports filtering by working directory and
     * cursor-based pagination.
     *
     * @experimental
     */
    unstable_listSessions(params: schema.ListSessionsRequest): Promise<schema.ListSessionsResponse>;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Resumes an existing session without returning previous messages.
     *
     * This method is only available if the agent advertises the `session.resume` capability.
     *
     * The agent should resume the session context, allowing the conversation to continue
     * without replaying the message history (unlike `session/load`).
     *
     * @experimental
     */
    unstable_resumeSession(params: schema.ResumeSessionRequest): Promise<schema.ResumeSessionResponse>;
    /**
     * Sets the operational mode for a session.
     *
     * Allows switching between different agent modes (e.g., "ask", "architect", "code")
     * that affect system prompts, tool availability, and permission behaviors.
     *
     * The mode must be one of the modes advertised in `availableModes` during session
     * creation or loading. Agents may also change modes autonomously and notify the
     * client via `current_mode_update` notifications.
     *
     * This method can be called at any time during a session, whether the Agent is
     * idle or actively generating a turn.
     *
     * See protocol docs: [Session Modes](https://agentclientprotocol.com/protocol/session-modes)
     */
    setSessionMode(params: schema.SetSessionModeRequest): Promise<schema.SetSessionModeResponse>;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Select a model for a given session.
     *
     * @experimental
     */
    unstable_setSessionModel(params: schema.SetSessionModelRequest): Promise<schema.SetSessionModelResponse>;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Set a configuration option for a given session.
     *
     * The response contains the full set of configuration options and their current values,
     * as changing one option may affect the available values or state of other options.
     *
     * @experimental
     */
    unstable_setSessionConfigOption(params: schema.SetSessionConfigOptionRequest): Promise<schema.SetSessionConfigOptionResponse>;
    /**
     * Authenticates the client using the specified authentication method.
     *
     * Called when the agent requires authentication before allowing session creation.
     * The client provides the authentication method ID that was advertised during initialization.
     *
     * After successful authentication, the client can proceed to create sessions with
     * `newSession` without receiving an `auth_required` error.
     *
     * See protocol docs: [Initialization](https://agentclientprotocol.com/protocol/initialization)
     */
    authenticate(params: schema.AuthenticateRequest): Promise<schema.AuthenticateResponse>;
    /**
     * Processes a user prompt within a session.
     *
     * This method handles the whole lifecycle of a prompt:
     * - Receives user messages with optional context (files, images, etc.)
     * - Processes the prompt using language models
     * - Reports language model content and tool calls to the Clients
     * - Requests permission to run tools
     * - Executes any requested tool calls
     * - Returns when the turn is complete with a stop reason
     *
     * See protocol docs: [Prompt Turn](https://agentclientprotocol.com/protocol/prompt-turn)
     */
    prompt(params: schema.PromptRequest): Promise<schema.PromptResponse>;
    /**
     * Cancels ongoing operations for a session.
     *
     * This is a notification sent by the client to cancel an ongoing prompt turn.
     *
     * Upon receiving this notification, the Agent SHOULD:
     * - Stop all language model requests as soon as possible
     * - Abort all tool call invocations in progress
     * - Send any pending `session/update` notifications
     * - Respond to the original `session/prompt` request with `StopReason::Cancelled`
     *
     * See protocol docs: [Cancellation](https://agentclientprotocol.com/protocol/prompt-turn#cancellation)
     */
    cancel(params: schema.CancelNotification): Promise<void>;
    /**
     * Extension method
     *
     * Allows the Client to send an arbitrary request that is not part of the ACP spec.
     */
    extMethod(method: string, params: Record<string, unknown>): Promise<Record<string, unknown>>;
    /**
     * Extension notification
     *
     * Allows the Client to send an arbitrary notification that is not part of the ACP spec.
     */
    extNotification(method: string, params: Record<string, unknown>): Promise<void>;
    /**
     * AbortSignal that aborts when the connection closes.
     *
     * This signal can be used to:
     * - Listen for connection closure: `connection.signal.addEventListener('abort', () => {...})`
     * - Check connection status synchronously: `if (connection.signal.aborted) {...}`
     * - Pass to other APIs (fetch, setTimeout) for automatic cancellation
     *
     * The connection closes when the underlying stream ends, either normally or due to an error.
     *
     * @example
     * ```typescript
     * const connection = new ClientSideConnection(client, stream);
     *
     * // Listen for closure
     * connection.signal.addEventListener('abort', () => {
     *   console.log('Connection closed - performing cleanup');
     * });
     *
     * // Check status
     * if (connection.signal.aborted) {
     *   console.log('Connection is already closed');
     * }
     *
     * // Pass to other APIs
     * fetch(url, { signal: connection.signal });
     * ```
     */
    get signal(): AbortSignal;
    /**
     * Promise that resolves when the connection closes.
     *
     * The connection closes when the underlying stream ends, either normally or due to an error.
     * Once closed, the connection cannot send or receive any more messages.
     *
     * This is useful for async/await style cleanup:
     *
     * @example
     * ```typescript
     * const connection = new ClientSideConnection(client, stream);
     * await connection.closed;
     * console.log('Connection closed - performing cleanup');
     * ```
     */
    get closed(): Promise<void>;
}
export type { AnyMessage } from "./jsonrpc.js";
declare class Connection {
    #private;
    constructor(requestHandler: RequestHandler, notificationHandler: NotificationHandler, stream: Stream);
    /**
     * AbortSignal that aborts when the connection closes.
     *
     * This signal can be used to:
     * - Listen for connection closure via event listeners
     * - Check connection status synchronously with `signal.aborted`
     * - Pass to other APIs (fetch, setTimeout) for automatic cancellation
     */
    get signal(): AbortSignal;
    /**
     * Promise that resolves when the connection closes.
     *
     * The connection closes when the underlying stream ends, either normally
     * or due to an error. Once closed, the connection cannot send or receive
     * any more messages.
     *
     * @example
     * ```typescript
     * const connection = new ClientSideConnection(client, stream);
     * await connection.closed;
     * console.log('Connection closed - performing cleanup');
     * ```
     */
    get closed(): Promise<void>;
    sendRequest<Req, Resp>(method: string, params?: Req): Promise<Resp>;
    sendNotification<N>(method: string, params?: N): Promise<void>;
}
/**
 * JSON-RPC error object.
 *
 * Represents an error that occurred during method execution, following the
 * JSON-RPC 2.0 error object specification with optional additional data.
 *
 * See protocol docs: [JSON-RPC Error Object](https://www.jsonrpc.org/specification#error_object)
 */
export declare class RequestError extends Error {
    code: number;
    data?: unknown;
    constructor(code: number, message: string, data?: unknown);
    /**
     * Invalid JSON was received by the server. An error occurred on the server while parsing the JSON text.
     */
    static parseError(data?: unknown, additionalMessage?: string): RequestError;
    /**
     * The JSON sent is not a valid Request object.
     */
    static invalidRequest(data?: unknown, additionalMessage?: string): RequestError;
    /**
     * The method does not exist / is not available.
     */
    static methodNotFound(method: string): RequestError;
    /**
     * Invalid method parameter(s).
     */
    static invalidParams(data?: unknown, additionalMessage?: string): RequestError;
    /**
     * Internal JSON-RPC error.
     */
    static internalError(data?: unknown, additionalMessage?: string): RequestError;
    /**
     * Authentication required.
     */
    static authRequired(data?: unknown, additionalMessage?: string): RequestError;
    /**
     * Resource, such as a file, was not found
     */
    static resourceNotFound(uri?: string): RequestError;
    toResult<T>(): Result<T>;
    toErrorResponse(): ErrorResponse;
}
/**
 * The Client interface defines the interface that ACP-compliant clients must implement.
 *
 * Clients are typically code editors (IDEs, text editors) that provide the interface
 * between users and AI agents. They manage the environment, handle user interactions,
 * and control access to resources.
 */
export interface Client {
    /**
     * Requests permission from the user for a tool call operation.
     *
     * Called by the agent when it needs user authorization before executing
     * a potentially sensitive operation. The client should present the options
     * to the user and return their decision.
     *
     * If the client cancels the prompt turn via `session/cancel`, it MUST
     * respond to this request with `RequestPermissionOutcome::Cancelled`.
     *
     * See protocol docs: [Requesting Permission](https://agentclientprotocol.com/protocol/tool-calls#requesting-permission)
     */
    requestPermission(params: schema.RequestPermissionRequest): Promise<schema.RequestPermissionResponse>;
    /**
     * Handles session update notifications from the agent.
     *
     * This is a notification endpoint (no response expected) that receives
     * real-time updates about session progress, including message chunks,
     * tool calls, and execution plans.
     *
     * Note: Clients SHOULD continue accepting tool call updates even after
     * sending a `session/cancel` notification, as the agent may send final
     * updates before responding with the cancelled stop reason.
     *
     * See protocol docs: [Agent Reports Output](https://agentclientprotocol.com/protocol/prompt-turn#3-agent-reports-output)
     */
    sessionUpdate(params: schema.SessionNotification): Promise<void>;
    /**
     * Writes content to a text file in the client's file system.
     *
     * Only available if the client advertises the `fs.writeTextFile` capability.
     * Allows the agent to create or modify files within the client's environment.
     *
     * See protocol docs: [Client](https://agentclientprotocol.com/protocol/overview#client)
     */
    writeTextFile?(params: schema.WriteTextFileRequest): Promise<schema.WriteTextFileResponse>;
    /**
     * Reads content from a text file in the client's file system.
     *
     * Only available if the client advertises the `fs.readTextFile` capability.
     * Allows the agent to access file contents within the client's environment.
     *
     * See protocol docs: [Client](https://agentclientprotocol.com/protocol/overview#client)
     */
    readTextFile?(params: schema.ReadTextFileRequest): Promise<schema.ReadTextFileResponse>;
    /**
     * Creates a new terminal to execute a command.
     *
     * Only available if the `terminal` capability is set to `true`.
     *
     * The Agent must call `releaseTerminal` when done with the terminal
     * to free resources.
  
     * @see {@link https://agentclientprotocol.com/protocol/terminals | Terminal Documentation}
     */
    createTerminal?(params: schema.CreateTerminalRequest): Promise<schema.CreateTerminalResponse>;
    /**
     * Gets the current output and exit status of a terminal.
     *
     * Returns immediately without waiting for the command to complete.
     * If the command has already exited, the exit status is included.
     *
     * @see {@link https://agentclientprotocol.com/protocol/terminals#getting-output | Getting Terminal Output}
     */
    terminalOutput?(params: schema.TerminalOutputRequest): Promise<schema.TerminalOutputResponse>;
    /**
     * Releases a terminal and frees all associated resources.
     *
     * The command is killed if it hasn't exited yet. After release,
     * the terminal ID becomes invalid for all other terminal methods.
     *
     * Tool calls that already contain the terminal ID continue to
     * display its output.
     *
     * @see {@link https://agentclientprotocol.com/protocol/terminals#releasing-terminals | Releasing Terminals}
     */
    releaseTerminal?(params: schema.ReleaseTerminalRequest): Promise<schema.ReleaseTerminalResponse | void>;
    /**
     * Waits for a terminal command to exit and returns its exit status.
     *
     * This method returns once the command completes, providing the
     * exit code and/or signal that terminated the process.
     *
     * @see {@link https://agentclientprotocol.com/protocol/terminals#waiting-for-exit | Waiting for Exit}
     */
    waitForTerminalExit?(params: schema.WaitForTerminalExitRequest): Promise<schema.WaitForTerminalExitResponse>;
    /**
     * Kills a terminal command without releasing the terminal.
     *
     * While `releaseTerminal` also kills the command, this method keeps
     * the terminal ID valid so it can be used with other methods.
     *
     * Useful for implementing command timeouts that terminate the command
     * and then retrieve the final output.
     *
     * Note: Call `releaseTerminal` when the terminal is no longer needed.
     *
     * @see {@link https://agentclientprotocol.com/protocol/terminals#killing-commands | Killing Commands}
     */
    killTerminal?(params: schema.KillTerminalCommandRequest): Promise<schema.KillTerminalCommandResponse | void>;
    /**
     * Extension method
     *
     * Allows the Agent to send an arbitrary request that is not part of the ACP spec.
     *
     * To help avoid conflicts, it's a good practice to prefix extension
     * methods with a unique identifier such as domain name.
     */
    extMethod?(method: string, params: Record<string, unknown>): Promise<Record<string, unknown>>;
    /**
     * Extension notification
     *
     * Allows the Agent to send an arbitrary notification that is not part of the ACP spec.
     */
    extNotification?(method: string, params: Record<string, unknown>): Promise<void>;
}
/**
 * The Agent interface defines the interface that all ACP-compliant agents must implement.
 *
 * Agents are programs that use generative AI to autonomously modify code. They handle
 * requests from clients and execute tasks using language models and tools.
 */
export interface Agent {
    /**
     * Establishes the connection with a client and negotiates protocol capabilities.
     *
     * This method is called once at the beginning of the connection to:
     * - Negotiate the protocol version to use
     * - Exchange capability information between client and agent
     * - Determine available authentication methods
     *
     * The agent should respond with its supported protocol version and capabilities.
     *
     * See protocol docs: [Initialization](https://agentclientprotocol.com/protocol/initialization)
     */
    initialize(params: schema.InitializeRequest): Promise<schema.InitializeResponse>;
    /**
     * Creates a new conversation session with the agent.
     *
     * Sessions represent independent conversation contexts with their own history and state.
     *
     * The agent should:
     * - Create a new session context
     * - Connect to any specified MCP servers
     * - Return a unique session ID for future requests
     *
     * May return an `auth_required` error if the agent requires authentication.
     *
     * See protocol docs: [Session Setup](https://agentclientprotocol.com/protocol/session-setup)
     */
    newSession(params: schema.NewSessionRequest): Promise<schema.NewSessionResponse>;
    /**
     * Loads an existing session to resume a previous conversation.
     *
     * This method is only available if the agent advertises the `loadSession` capability.
     *
     * The agent should:
     * - Restore the session context and conversation history
     * - Connect to the specified MCP servers
     * - Stream the entire conversation history back to the client via notifications
     *
     * See protocol docs: [Loading Sessions](https://agentclientprotocol.com/protocol/session-setup#loading-sessions)
     */
    loadSession?(params: schema.LoadSessionRequest): Promise<schema.LoadSessionResponse>;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Forks an existing session to create a new independent session.
     *
     * Creates a new session based on the context of an existing one, allowing
     * operations like generating summaries without affecting the original session's history.
     *
     * This method is only available if the agent advertises the `session.fork` capability.
     *
     * @experimental
     */
    unstable_forkSession?(params: schema.ForkSessionRequest): Promise<schema.ForkSessionResponse>;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Lists existing sessions from the agent.
     *
     * This method is only available if the agent advertises the `listSessions` capability.
     *
     * Returns a list of sessions with metadata like session ID, working directory,
     * title, and last update time. Supports filtering by working directory and
     * cursor-based pagination.
     */
    unstable_listSessions?(params: schema.ListSessionsRequest): Promise<schema.ListSessionsResponse>;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Resumes an existing session without returning previous messages.
     *
     * This method is only available if the agent advertises the `session.resume` capability.
     *
     * The agent should resume the session context, allowing the conversation to continue
     * without replaying the message history (unlike `session/load`).
     *
     * @experimental
     */
    unstable_resumeSession?(params: schema.ResumeSessionRequest): Promise<schema.ResumeSessionResponse>;
    /**
     * Sets the operational mode for a session.
     *
     * Allows switching between different agent modes (e.g., "ask", "architect", "code")
     * that affect system prompts, tool availability, and permission behaviors.
     *
     * The mode must be one of the modes advertised in `availableModes` during session
     * creation or loading. Agents may also change modes autonomously and notify the
     * client via `current_mode_update` notifications.
     *
     * This method can be called at any time during a session, whether the Agent is
     * idle or actively generating a turn.
     *
     * See protocol docs: [Session Modes](https://agentclientprotocol.com/protocol/session-modes)
     */
    setSessionMode?(params: schema.SetSessionModeRequest): Promise<schema.SetSessionModeResponse | void>;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Select a model for a given session.
     *
     * @experimental
     */
    unstable_setSessionModel?(params: schema.SetSessionModelRequest): Promise<schema.SetSessionModelResponse | void>;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Set a configuration option for a given session.
     *
     * The response contains the full set of configuration options and their current values,
     * as changing one option may affect the available values or state of other options.
     *
     * @experimental
     */
    unstable_setSessionConfigOption?(params: schema.SetSessionConfigOptionRequest): Promise<schema.SetSessionConfigOptionResponse>;
    /**
     * Authenticates the client using the specified authentication method.
     *
     * Called when the agent requires authentication before allowing session creation.
     * The client provides the authentication method ID that was advertised during initialization.
     *
     * After successful authentication, the client can proceed to create sessions with
     * `newSession` without receiving an `auth_required` error.
     *
     * See protocol docs: [Initialization](https://agentclientprotocol.com/protocol/initialization)
     */
    authenticate(params: schema.AuthenticateRequest): Promise<schema.AuthenticateResponse | void>;
    /**
     * Processes a user prompt within a session.
     *
     * This method handles the whole lifecycle of a prompt:
     * - Receives user messages with optional context (files, images, etc.)
     * - Processes the prompt using language models
     * - Reports language model content and tool calls to the Clients
     * - Requests permission to run tools
     * - Executes any requested tool calls
     * - Returns when the turn is complete with a stop reason
     *
     * See protocol docs: [Prompt Turn](https://agentclientprotocol.com/protocol/prompt-turn)
     */
    prompt(params: schema.PromptRequest): Promise<schema.PromptResponse>;
    /**
     * Cancels ongoing operations for a session.
     *
     * This is a notification sent by the client to cancel an ongoing prompt turn.
     *
     * Upon receiving this notification, the Agent SHOULD:
     * - Stop all language model requests as soon as possible
     * - Abort all tool call invocations in progress
     * - Send any pending `session/update` notifications
     * - Respond to the original `session/prompt` request with `StopReason::Cancelled`
     *
     * See protocol docs: [Cancellation](https://agentclientprotocol.com/protocol/prompt-turn#cancellation)
     */
    cancel(params: schema.CancelNotification): Promise<void>;
    /**
     * Extension method
     *
     * Allows the Client to send an arbitrary request that is not part of the ACP spec.
     *
     * To help avoid conflicts, it's a good practice to prefix extension
     * methods with a unique identifier such as domain name.
     */
    extMethod?(method: string, params: Record<string, unknown>): Promise<Record<string, unknown>>;
    /**
     * Extension notification
     *
     * Allows the Client to send an arbitrary notification that is not part of the ACP spec.
     */
    extNotification?(method: string, params: Record<string, unknown>): Promise<void>;
}
