import { type Api, type Other as OtherApi } from "./core/api.js";
import { type Methods, type RawApi } from "./core/client.js";
import { type Filter, type FilterCore, type FilterQuery } from "./filter.js";
import { type AcceptedGiftTypes, type Chat, type ChatPermissions, type InlineQueryResult, type InputChecklist, type InputFile, type InputMedia, type InputMediaAudio, type InputMediaDocument, type InputMediaPhoto, type InputMediaVideo, type InputPaidMedia, type InputPollOption, type InputProfilePhoto, type InputStoryContent, type LabeledPrice, type Message, type MessageEntity, type PassportElementError, type ReactionType, type ReactionTypeEmoji, type Update, type User, type UserFromGetMe } from "./types.js";
export type MaybeArray<T> = T | T[];
/** permits `string` but gives hints */
export type StringWithCommandSuggestions = (string & Record<never, never>) | "start" | "help" | "settings" | "privacy" | "developer_info";
type Other<M extends Methods<RawApi>, X extends string = never> = OtherApi<RawApi, M, X>;
type SnakeToCamelCase<S extends string> = S extends `${infer L}_${infer R}` ? `${L}${Capitalize<SnakeToCamelCase<R>>}` : S;
type AliasProps<U> = {
    [K in string & keyof U as SnakeToCamelCase<K>]: U[K];
};
type RenamedUpdate = AliasProps<Omit<Update, "update_id">>;
interface StaticHas {
    /**
     * Generates a predicate function that can test context objects for matching
     * the given filter query. This uses the same logic as `bot.on`.
     *
     * @param filter The filter query to check
     */
    filterQuery<Q extends FilterQuery>(filter: Q | Q[]): <C extends Context>(ctx: C) => ctx is Filter<C, Q>;
    /**
     * Generates a predicate function that can test context objects for
     * containing the given text, or for the text to match the given regular
     * expression. This uses the same logic as `bot.hears`.
     *
     * @param trigger The string or regex to match
     */
    text(trigger: MaybeArray<string | RegExp>): <C extends Context>(ctx: C) => ctx is HearsContext<C>;
    /**
     * Generates a predicate function that can test context objects for
     * containing a command. This uses the same logic as `bot.command`.
     *
     * @param command The command to match
     */
    command(command: MaybeArray<StringWithCommandSuggestions>): <C extends Context>(ctx: C) => ctx is CommandContext<C>;
    /**
     * Generates a predicate function that can test context objects for
     * containing a message reaction update. This uses the same logic as
     * `bot.reaction`.
     *
     * @param reaction The reaction to test against
     */
    reaction(reaction: MaybeArray<ReactionTypeEmoji["emoji"] | ReactionType>): <C extends Context>(ctx: C) => ctx is ReactionContext<C>;
    /**
     * Generates a predicate function that can test context objects for
     * belonging to a chat with the given chat type. This uses the same logic as
     * `bot.chatType`.
     *
     * @param chatType The chat type to match
     */
    chatType<T extends Chat["type"]>(chatType: MaybeArray<T>): <C extends Context>(ctx: C) => ctx is ChatTypeContext<C, T>;
    /**
     * Generates a predicate function that can test context objects for
     * containing the given callback query, or for the callback query data to
     * match the given regular expression. This uses the same logic as
     * `bot.callbackQuery`.
     *
     * @param trigger The string or regex to match
     */
    callbackQuery(trigger: MaybeArray<string | RegExp>): <C extends Context>(ctx: C) => ctx is CallbackQueryContext<C>;
    /**
     * Generates a predicate function that can test context objects for
     * containing the given game query, or for the game name to match the given
     * regular expression. This uses the same logic as `bot.gameQuery`.
     *
     * @param trigger The string or regex to match
     */
    gameQuery(trigger: MaybeArray<string | RegExp>): <C extends Context>(ctx: C) => ctx is GameQueryContext<C>;
    /**
     * Generates a predicate function that can test context objects for
     * containing the given inline query, or for the inline query to match the
     * given regular expression. This uses the same logic as `bot.inlineQuery`.
     *
     * @param trigger The string or regex to match
     */
    inlineQuery(trigger: MaybeArray<string | RegExp>): <C extends Context>(ctx: C) => ctx is InlineQueryContext<C>;
    /**
     * Generates a predicate function that can test context objects for
     * containing the chosen inline result, or for the chosen inline result to
     * match the given regular expression.
     *
     * @param trigger The string or regex to match
     */
    chosenInlineResult(trigger: MaybeArray<string | RegExp>): <C extends Context>(ctx: C) => ctx is ChosenInlineResultContext<C>;
    /**
     * Generates a predicate function that can test context objects for
     * containing the given pre-checkout query, or for the pre-checkout query
     * payload to match the given regular expression. This uses the same logic
     * as `bot.preCheckoutQuery`.
     *
     * @param trigger The string or regex to match
     */
    preCheckoutQuery(trigger: MaybeArray<string | RegExp>): <C extends Context>(ctx: C) => ctx is PreCheckoutQueryContext<C>;
    /**
     * Generates a predicate function that can test context objects for
     * containing the given shipping query, or for the shipping query to match
     * the given regular expression. This uses the same logic as
     * `bot.shippingQuery`.
     *
     * @param trigger The string or regex to match
     */
    shippingQuery(trigger: MaybeArray<string | RegExp>): <C extends Context>(ctx: C) => ctx is ShippingQueryContext<C>;
}
/**
 * When your bot receives a message, Telegram sends an update object to your
 * bot. The update contains information about the chat, the user, and of course
 * the message itself. There are numerous other updates, too:
 * https://core.telegram.org/bots/api#update
 *
 * When grammY receives an update, it wraps this update into a context object
 * for you. Context objects are commonly named `ctx`. A context object does two
 * things:
 * 1. **`ctx.update`** holds the update object that you can use to process the
 *    message. This includes providing useful shortcuts for the update, for
 *    instance, `ctx.msg` is a shortcut that gives you the message object from
 *    the update—no matter whether it is contained in `ctx.update.message`, or
 *    `ctx.update.edited_message`, or `ctx.update.channel_post`, or
 *    `ctx.update.edited_channel_post`.
 * 2. **`ctx.api`** gives you access to the full Telegram Bot API so that you
 *    can directly call any method, such as responding via
 *    `ctx.api.sendMessage`. Also here, the context objects has some useful
 *    shortcuts for you. For instance, if you want to send a message to the same
 *    chat that a message comes from (i.e. just respond to a user) you can call
 *    `ctx.reply`. This is nothing but a wrapper for `ctx.api.sendMessage` with
 *    the right `chat_id` pre-filled for you. Almost all methods of the Telegram
 *    Bot API have their own shortcut directly on the context object, so you
 *    probably never really have to use `ctx.api` at all.
 *
 * This context object is then passed to all of the listeners (called
 * middleware) that you register on your bot. Because this is so useful, the
 * context object is often used to hold more information. One example are
 * sessions (a chat-specific data storage that is stored in a database), and
 * another example is `ctx.match` that is used by `bot.command` and other
 * methods to keep information about how a regular expression was matched.
 *
 * Read up about middleware on the
 * [website](https://grammy.dev/guide/context) if you want to know more
 * about the powerful opportunities that lie in context objects, and about how
 * grammY implements them.
 */
export declare class Context implements RenamedUpdate {
    /**
     * The update object that is contained in the context.
     */
    readonly update: Update;
    /**
     * An API instance that allows you to call any method of the Telegram
     * Bot API.
     */
    readonly api: Api;
    /**
     * Information about the bot itself.
     */
    readonly me: UserFromGetMe;
    /**
     * Used by some middleware to store information about how a certain string
     * or regular expression was matched.
     */
    match: string | RegExpMatchArray | undefined;
    constructor(
    /**
     * The update object that is contained in the context.
     */
    update: Update, 
    /**
     * An API instance that allows you to call any method of the Telegram
     * Bot API.
     */
    api: Api, 
    /**
     * Information about the bot itself.
     */
    me: UserFromGetMe);
    /** Alias for `ctx.update.message` */
    get message(): (Message & Update.NonChannel) | undefined;
    /** Alias for `ctx.update.edited_message` */
    get editedMessage(): (Message & Update.Edited & Update.NonChannel) | undefined;
    /** Alias for `ctx.update.channel_post` */
    get channelPost(): (Message & Update.Channel) | undefined;
    /** Alias for `ctx.update.edited_channel_post` */
    get editedChannelPost(): (Message & Update.Edited & Update.Channel) | undefined;
    /** Alias for `ctx.update.business_connection` */
    get businessConnection(): import("@grammyjs/types/manage.js").BusinessConnection | undefined;
    /** Alias for `ctx.update.business_message` */
    get businessMessage(): (Message & Update.Private) | undefined;
    /** Alias for `ctx.update.edited_business_message` */
    get editedBusinessMessage(): (Message & Update.Edited & Update.Private) | undefined;
    /** Alias for `ctx.update.deleted_business_messages` */
    get deletedBusinessMessages(): import("@grammyjs/types/manage.js").BusinessMessagesDeleted | undefined;
    /** Alias for `ctx.update.message_reaction` */
    get messageReaction(): import("@grammyjs/types/message.js").MessageReactionUpdated | undefined;
    /** Alias for `ctx.update.message_reaction_count` */
    get messageReactionCount(): import("@grammyjs/types/message.js").MessageReactionCountUpdated | undefined;
    /** Alias for `ctx.update.inline_query` */
    get inlineQuery(): import("@grammyjs/types/inline.js").InlineQuery | undefined;
    /** Alias for `ctx.update.chosen_inline_result` */
    get chosenInlineResult(): import("@grammyjs/types/inline.js").ChosenInlineResult | undefined;
    /** Alias for `ctx.update.callback_query` */
    get callbackQuery(): import("@grammyjs/types/markup.js").CallbackQuery | undefined;
    /** Alias for `ctx.update.shipping_query` */
    get shippingQuery(): import("@grammyjs/types/payment.js").ShippingQuery | undefined;
    /** Alias for `ctx.update.pre_checkout_query` */
    get preCheckoutQuery(): import("@grammyjs/types/payment.js").PreCheckoutQuery | undefined;
    /** Alias for `ctx.update.poll` */
    get poll(): import("@grammyjs/types/message.js").Poll | undefined;
    /** Alias for `ctx.update.poll_answer` */
    get pollAnswer(): import("@grammyjs/types/message.js").PollAnswer | undefined;
    /** Alias for `ctx.update.my_chat_member` */
    get myChatMember(): import("@grammyjs/types/manage.js").ChatMemberUpdated | undefined;
    /** Alias for `ctx.update.chat_member` */
    get chatMember(): import("@grammyjs/types/manage.js").ChatMemberUpdated | undefined;
    /** Alias for `ctx.update.chat_join_request` */
    get chatJoinRequest(): import("@grammyjs/types/manage.js").ChatJoinRequest | undefined;
    /** Alias for `ctx.update.chat_boost` */
    get chatBoost(): import("@grammyjs/types/manage.js").ChatBoostUpdated | undefined;
    /** Alias for `ctx.update.removed_chat_boost` */
    get removedChatBoost(): import("@grammyjs/types/manage.js").ChatBoostRemoved | undefined;
    /** Alias for `ctx.update.purchased_paid_media` */
    get purchasedPaidMedia(): import("@grammyjs/types/payment.js").PaidMediaPurchased | undefined;
    /**
     * Get the message object from wherever possible. Alias for `this.message ??
     * this.editedMessage ?? this.channelPost ?? this.editedChannelPost ??
     * this.businessMessage ?? this.editedBusinessMessage ??
     * this.callbackQuery?.message`.
     */
    get msg(): Message | undefined;
    /**
     * Get the chat object from wherever possible. Alias for `(this.msg ??
     * this.deletedBusinessMessages ?? this.messageReaction ??
     * this.messageReactionCount ?? this.myChatMember ??  this.chatMember ??
     * this.chatJoinRequest ?? this.chatBoost ??  this.removedChatBoost)?.chat`.
     */
    get chat(): Chat | undefined;
    /**
     * Get the sender chat object from wherever possible. Alias for
     * `ctx.msg?.sender_chat`.
     */
    get senderChat(): Chat | undefined;
    /**
     * Get the user object from wherever possible. Alias for
     * `(this.businessConnection ?? this.messageReaction ??
     * (this.chatBoost?.boost ?? this.removedChatBoost)?.source)?.user ??
     * (this.callbackQuery ?? this.msg ?? this.inlineQuery ??
     * this.chosenInlineResult ?? this.shippingQuery ?? this.preCheckoutQuery ??
     * this.myChatMember ?? this.chatMember ?? this.chatJoinRequest ??
     * this.purchasedPaidMedia)?.from`.
     */
    get from(): User | undefined;
    /**
     * Get the message identifier from wherever possible. Alias for
     * `this.msg?.message_id ?? this.messageReaction?.message_id ??
     * this.messageReactionCount?.message_id`.
     */
    get msgId(): number | undefined;
    /**
     * Gets the chat identifier from wherever possible. Alias for `this.chat?.id
     * ?? this.businessConnection?.user_chat_id`.
     */
    get chatId(): number | undefined;
    /**
     * Get the inline message identifier from wherever possible. Alias for
     * `(ctx.callbackQuery ?? ctx.chosenInlineResult)?.inline_message_id`.
     */
    get inlineMessageId(): string | undefined;
    /**
     * Get the business connection identifier from wherever possible. Alias for
     * `this.msg?.business_connection_id ?? this.businessConnection?.id ??
     * this.deletedBusinessMessages?.business_connection_id`.
     */
    get businessConnectionId(): string | undefined;
    /**
     * Get entities and their text. Extracts the text from `ctx.msg.text` or
     * `ctx.msg.caption`. Returns an empty array if one of `ctx.msg`,
     * `ctx.msg.text` or `ctx.msg.entities` is undefined.
     *
     * You can filter specific entity types by passing the `types` parameter.
     * Example:
     *
     * ```ts
     * ctx.entities() // Returns all entity types
     * ctx.entities('url') // Returns only url entities
     * ctx.entities(['url', 'email']) // Returns url and email entities
     * ```
     *
     * @param types Types of entities to return. Omit to get all entities.
     * @returns Array of entities and their texts, or empty array when there's no text
     */
    entities(): Array<MessageEntity & {
        /** Slice of the message text that contains this entity */
        text: string;
    }>;
    entities<T extends MessageEntity["type"]>(types: MaybeArray<T>): Array<MessageEntity & {
        type: T;
        /** Slice of the message text that contains this entity */
        text: string;
    }>;
    /**
     * Find out which reactions were added and removed in a `message_reaction`
     * update. This method looks at `ctx.messageReaction` and computes the
     * difference between the old reaction and the new reaction. It also groups
     * the reactions by emoji reactions and custom emoji reactions. For example,
     * the resulting object could look like this:
     * ```ts
     * {
     *   emoji: ['👍', '🎉']
     *   emojiAdded: ['🎉'],
     *   emojiKept: ['👍'],
     *   emojiRemoved: [],
     *   customEmoji: [],
     *   customEmojiAdded: [],
     *   customEmojiKept: [],
     *   customEmojiRemoved: ['id0123'],
     *   paid: true,
     *   paidAdded: false,
     *   paidRemoved: false,
     * }
     * ```
     * In the above example, a tada reaction was added by the user, and a custom
     * emoji reaction with the custom emoji 'id0123' was removed in the same
     * update. The user had already reacted with a thumbs up reaction and a paid
     * star reaction, which they left both unchanged. As a result, the current
     * reaction by the user is thumbs up, tada, and a paid reaction. Note that
     * the current reaction (all emoji reactions regardless of type in one list)
     * can also be obtained from `ctx.messageReaction.new_reaction`.
     *
     * Remember that reaction updates only include information about the
     * reaction of a specific user. The respective message may have many more
     * reactions by other people which will not be included in this update.
     *
     * @returns An object containing information about the reaction update
     */
    reactions(): {
        /** Emoji currently present in this user's reaction */
        emoji: ReactionTypeEmoji["emoji"][];
        /** Emoji newly added to this user's reaction */
        emojiAdded: ReactionTypeEmoji["emoji"][];
        /** Emoji not changed by the update to this user's reaction */
        emojiKept: ReactionTypeEmoji["emoji"][];
        /** Emoji removed from this user's reaction */
        emojiRemoved: ReactionTypeEmoji["emoji"][];
        /** Custom emoji currently present in this user's reaction */
        customEmoji: string[];
        /** Custom emoji newly added to this user's reaction */
        customEmojiAdded: string[];
        /** Custom emoji not changed by the update to this user's reaction */
        customEmojiKept: string[];
        /** Custom emoji removed from this user's reaction */
        customEmojiRemoved: string[];
        /**
         * `true` if a paid reaction is currently present in this user's
         * reaction, and `false` otherwise
         */
        paid: boolean;
        /**
         * `true` if a paid reaction was newly added to this user's reaction,
         * and `false` otherwise
         */
        paidAdded: boolean;
    };
    /**
     * `Context.has` is an object that contains a number of useful functions for
     * probing context objects. Each of these functions can generate a predicate
     * function, to which you can pass context objects in order to check if a
     * condition holds for the respective context object.
     *
     * For example, you can call `Context.has.filterQuery(":text")` to generate
     * a predicate function that tests context objects for containing text:
     * ```ts
     * const hasText = Context.has.filterQuery(":text");
     *
     * if (hasText(ctx0)) {} // `ctx0` matches the filter query `:text`
     * if (hasText(ctx1)) {} // `ctx1` matches the filter query `:text`
     * if (hasText(ctx2)) {} // `ctx2` matches the filter query `:text`
     * ```
     * These predicate functions are used internally by the has-methods that are
     * installed on every context object. This means that calling
     * `ctx.has(":text")` is equivalent to
     * `Context.has.filterQuery(":text")(ctx)`.
     */
    static has: StaticHas;
    /**
     * Returns `true` if this context object matches the given filter query, and
     * `false` otherwise. This uses the same logic as `bot.on`.
     *
     * @param filter The filter query to check
     */
    has<Q extends FilterQuery>(filter: Q | Q[]): this is FilterCore<Q>;
    /**
     * Returns `true` if this context object contains the given text, or if it
     * contains text that matches the given regular expression. It returns
     * `false` otherwise. This uses the same logic as `bot.hears`.
     *
     * @param trigger The string or regex to match
     */
    hasText(trigger: MaybeArray<string | RegExp>): this is HearsContextCore;
    /**
     * Returns `true` if this context object contains the given command, and
     * `false` otherwise. This uses the same logic as `bot.command`.
     *
     * @param command The command to match
     */
    hasCommand(command: MaybeArray<StringWithCommandSuggestions>): this is CommandContextCore;
    hasReaction(reaction: MaybeArray<ReactionTypeEmoji["emoji"] | ReactionType>): this is ReactionContextCore;
    /**
     * Returns `true` if this context object belongs to a chat with the given
     * chat type, and `false` otherwise. This uses the same logic as
     * `bot.chatType`.
     *
     * @param chatType The chat type to match
     */
    hasChatType<T extends Chat["type"]>(chatType: MaybeArray<T>): this is ChatTypeContextCore<T>;
    /**
     * Returns `true` if this context object contains the given callback query,
     * or if the contained callback query data matches the given regular
     * expression. It returns `false` otherwise. This uses the same logic as
     * `bot.callbackQuery`.
     *
     * @param trigger The string or regex to match
     */
    hasCallbackQuery(trigger: MaybeArray<string | RegExp>): this is CallbackQueryContextCore;
    /**
     * Returns `true` if this context object contains the given game query, or
     * if the contained game query matches the given regular expression. It
     * returns `false` otherwise. This uses the same logic as `bot.gameQuery`.
     *
     * @param trigger The string or regex to match
     */
    hasGameQuery(trigger: MaybeArray<string | RegExp>): this is GameQueryContextCore;
    /**
     * Returns `true` if this context object contains the given inline query, or
     * if the contained inline query matches the given regular expression. It
     * returns `false` otherwise. This uses the same logic as `bot.inlineQuery`.
     *
     * @param trigger The string or regex to match
     */
    hasInlineQuery(trigger: MaybeArray<string | RegExp>): this is InlineQueryContextCore;
    /**
     * Returns `true` if this context object contains the chosen inline result,
     * or if the contained chosen inline result matches the given regular
     * expression. It returns `false` otherwise. This uses the same logic as
     * `bot.chosenInlineResult`.
     *
     * @param trigger The string or regex to match
     */
    hasChosenInlineResult(trigger: MaybeArray<string | RegExp>): this is ChosenInlineResultContextCore;
    /**
     * Returns `true` if this context object contains the given pre-checkout
     * query, or if the contained pre-checkout query matches the given regular
     * expression. It returns `false` otherwise. This uses the same logic as
     * `bot.preCheckoutQuery`.
     *
     * @param trigger The string or regex to match
     */
    hasPreCheckoutQuery(trigger: MaybeArray<string | RegExp>): this is PreCheckoutQueryContextCore;
    /**
     * Returns `true` if this context object contains the given shipping query,
     * or if the contained shipping query matches the given regular expression.
     * It returns `false` otherwise. This uses the same logic as
     * `bot.shippingQuery`.
     *
     * @param trigger The string or regex to match
     */
    hasShippingQuery(trigger: MaybeArray<string | RegExp>): this is ShippingQueryContextCore;
    /**
     * Context-aware alias for `api.sendMessage`. Use this method to send text messages. On success, the sent Message is returned.
     *
     * @param text Text of the message to be sent, 1-4096 characters after entities parsing
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendmessage
     */
    reply(text: string, other?: Other<"sendMessage", "chat_id" | "text">, signal?: AbortSignal): Promise<Message.TextMessage>;
    /**
     * Context-aware alias for `api.sendMessageDraft`. Use this method to stream a partial message to a user while the message is being generated; supported only for bots with forum topic mode enabled. Returns True on success.
     *
     * @param text Text of the message to be sent, 1-4096 characters after entities parsing
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendmessagedraft
     */
    replyWithDraft(text: string, other?: Other<"sendMessageDraft", "chat_id" | "text">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.forwardMessage`. Use this method to forward messages of any kind. Service messages and messages with protected content can't be forwarded. On success, the sent Message is returned.
     *
     * @param chat_id Unique identifier for the target chat or username of the target channel (in the format @channelusername)
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#forwardmessage
     */
    forwardMessage(chat_id: number | string, other?: Other<"forwardMessage", "chat_id" | "from_chat_id" | "message_id">, signal?: AbortSignal): Promise<Message>;
    /**
     * Context-aware alias for `api.forwardMessages`. Use this method to forward multiple messages of any kind. If some of the specified messages can't be found or forwarded, they are skipped. Service messages and messages with protected content can't be forwarded. Album grouping is kept for forwarded messages. On success, an array of MessageId of the sent messages is returned.
     *
     * @param chat_id Unique identifier for the target chat or username of the target channel (in the format @channelusername)
     * @param message_ids A list of 1-100 identifiers of messages in the current chat to forward. The identifiers must be specified in a strictly increasing order.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#forwardmessages
     */
    forwardMessages(chat_id: number | string, message_ids: number[], other?: Other<"forwardMessages", "chat_id" | "from_chat_id" | "message_ids">, signal?: AbortSignal): Promise<import("@grammyjs/types/message.js").MessageId[]>;
    /**
     * Context-aware alias for `api.copyMessage`. Use this method to copy messages of any kind. Service messages, paid media messages, giveaway messages, giveaway winners messages, and invoice messages can't be copied. A quiz poll can be copied only if the value of the field correct_option_id is known to the bot. The method is analogous to the method forwardMessage, but the copied message doesn't have a link to the original message. Returns the MessageId of the sent message on success.
     *
     * @param chat_id Unique identifier for the target chat or username of the target channel (in the format @channelusername)
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#copymessage
     */
    copyMessage(chat_id: number | string, other?: Other<"copyMessage", "chat_id" | "from_chat_id" | "message_id">, signal?: AbortSignal): Promise<import("@grammyjs/types/message.js").MessageId>;
    /**
     * Context-aware alias for `api.copyMessages`. Use this method to copy messages of any kind. If some of the specified messages can't be found or copied, they are skipped. Service messages, paid media messages, giveaway messages, giveaway winners messages, and invoice messages can't be copied. A quiz poll can be copied only if the value of the field correct_option_id is known to the bot. The method is analogous to the method forwardMessages, but the copied messages don't have a link to the original message. Album grouping is kept for copied messages. On success, an array of MessageId of the sent messages is returned.
     *
     * @param chat_id Unique identifier for the target chat or username of the target channel (in the format @channelusername)
     * @param message_ids A list of 1-100 identifiers of messages in the current chat to copy. The identifiers must be specified in a strictly increasing order.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#copymessages
     */
    copyMessages(chat_id: number | string, message_ids: number[], other?: Other<"copyMessages", "chat_id" | "from_chat_id" | "message_ids">, signal?: AbortSignal): Promise<import("@grammyjs/types/message.js").MessageId[]>;
    /**
     * Context-aware alias for `api.sendPhoto`. Use this method to send photos. On success, the sent Message is returned.
     *
     * @param photo Photo to send. Pass a file_id as String to send a photo that exists on the Telegram servers (recommended), pass an HTTP URL as a String for Telegram to get a photo from the Internet, or upload a new photo using multipart/form-data. The photo must be at most 10 MB in size. The photo's width and height must not exceed 10000 in total. Width and height ratio must be at most 20.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendphoto
     */
    replyWithPhoto(photo: InputFile | string, other?: Other<"sendPhoto", "chat_id" | "photo">, signal?: AbortSignal): Promise<Message.PhotoMessage>;
    /**
     * Context-aware alias for `api.sendAudio`. Use this method to send audio files, if you want Telegram clients to display them in the music player. Your audio must be in the .MP3 or .M4A format. On success, the sent Message is returned. Bots can currently send audio files of up to 50 MB in size, this limit may be changed in the future.
     *
     * For sending voice messages, use the sendVoice method instead.
     *
     * @param audio Audio file to send. Pass a file_id as String to send an audio file that exists on the Telegram servers (recommended), pass an HTTP URL as a String for Telegram to get an audio file from the Internet, or upload a new one using multipart/form-data.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendaudio
     */
    replyWithAudio(audio: InputFile | string, other?: Other<"sendAudio", "chat_id" | "audio">, signal?: AbortSignal): Promise<Message.AudioMessage>;
    /**
     * Context-aware alias for `api.sendDocument`. Use this method to send general files. On success, the sent Message is returned. Bots can currently send files of any type of up to 50 MB in size, this limit may be changed in the future.
     *
     * @param document File to send. Pass a file_id as String to send a file that exists on the Telegram servers (recommended), pass an HTTP URL as a String for Telegram to get a file from the Internet, or upload a new one using multipart/form-data.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#senddocument
     */
    replyWithDocument(document: InputFile | string, other?: Other<"sendDocument", "chat_id" | "document">, signal?: AbortSignal): Promise<Message.DocumentMessage>;
    /**
     * Context-aware alias for `api.sendVideo`. Use this method to send video files, Telegram clients support mp4 videos (other formats may be sent as Document). On success, the sent Message is returned. Bots can currently send video files of up to 50 MB in size, this limit may be changed in the future.
     *
     * @param video Video to send. Pass a file_id as String to send a video that exists on the Telegram servers (recommended), pass an HTTP URL as a String for Telegram to get a video from the Internet, or upload a new video using multipart/form-data.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendvideo
     */
    replyWithVideo(video: InputFile | string, other?: Other<"sendVideo", "chat_id" | "video">, signal?: AbortSignal): Promise<Message.VideoMessage>;
    /**
     * Context-aware alias for `api.sendAnimation`. Use this method to send animation files (GIF or H.264/MPEG-4 AVC video without sound). On success, the sent Message is returned. Bots can currently send animation files of up to 50 MB in size, this limit may be changed in the future.
     *
     * @param animation Animation to send. Pass a file_id as String to send an animation that exists on the Telegram servers (recommended), pass an HTTP URL as a String for Telegram to get an animation from the Internet, or upload a new animation using multipart/form-data.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendanimation
     */
    replyWithAnimation(animation: InputFile | string, other?: Other<"sendAnimation", "chat_id" | "animation">, signal?: AbortSignal): Promise<Message.AnimationMessage>;
    /**
     * Context-aware alias for `api.sendVoice`. Use this method to send audio files, if you want Telegram clients to display the file as a playable voice message. For this to work, your audio must be in an .OGG file encoded with OPUS (other formats may be sent as Audio or Document). On success, the sent Message is returned. Bots can currently send voice messages of up to 50 MB in size, this limit may be changed in the future.
     *
     * @param voice Audio file to send. Pass a file_id as String to send a file that exists on the Telegram servers (recommended), pass an HTTP URL as a String for Telegram to get a file from the Internet, or upload a new one using multipart/form-data.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendvoice
     */
    replyWithVoice(voice: InputFile | string, other?: Other<"sendVoice", "chat_id" | "voice">, signal?: AbortSignal): Promise<Message.VoiceMessage>;
    /**
     * Context-aware alias for `api.sendVideoNote`. Use this method to send video messages. On success, the sent Message is returned.
     * As of v.4.0, Telegram clients support rounded square mp4 videos of up to 1 minute long.
     *
     * @param video_note Video note to send. Pass a file_id as String to send a video note that exists on the Telegram servers (recommended) or upload a new video using multipart/form-data.. Sending video notes by a URL is currently unsupported
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendvideonote
     */
    replyWithVideoNote(video_note: InputFile | string, other?: Other<"sendVideoNote", "chat_id" | "video_note">, signal?: AbortSignal): Promise<Message.VideoNoteMessage>;
    /**
     * Context-aware alias for `api.sendMediaGroup`. Use this method to send a group of photos, videos, documents or audios as an album. Documents and audio files can be only grouped in an album with messages of the same type. On success, an array of Messages that were sent is returned.
     *
     * @param media An array describing messages to be sent, must include 2-10 items
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendmediagroup
     */
    replyWithMediaGroup(media: ReadonlyArray<InputMediaAudio | InputMediaDocument | InputMediaPhoto | InputMediaVideo>, other?: Other<"sendMediaGroup", "chat_id" | "media">, signal?: AbortSignal): Promise<(Message.PhotoMessage | Message.AudioMessage | Message.DocumentMessage | Message.VideoMessage)[]>;
    /**
     * Context-aware alias for `api.sendLocation`. Use this method to send point on the map. On success, the sent Message is returned.
     *
     * @param latitude Latitude of the location
     * @param longitude Longitude of the location
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendlocation
     */
    replyWithLocation(latitude: number, longitude: number, other?: Other<"sendLocation", "chat_id" | "latitude" | "longitude">, signal?: AbortSignal): Promise<Message.LocationMessage>;
    /**
     * Context-aware alias for `api.editMessageLiveLocation`. Use this method to edit live location messages. A location can be edited until its live_period expires or editing is explicitly disabled by a call to stopMessageLiveLocation. On success, if the edited message is not an inline message, the edited Message is returned, otherwise True is returned.
     *
     * @param latitude Latitude of new location
     * @param longitude Longitude of new location
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#editmessagelivelocation
     */
    editMessageLiveLocation(latitude: number, longitude: number, other?: Other<"editMessageLiveLocation", "chat_id" | "message_id" | "inline_message_id" | "latitude" | "longitude">, signal?: AbortSignal): Promise<true | (Update.Edited & Message.CommonMessage & {
        location: import("@grammyjs/types/message.js").Location;
    })>;
    /**
     * Context-aware alias for `api.stopMessageLiveLocation`. Use this method to stop updating a live location message before live_period expires. On success, if the message is not an inline message, the edited Message is returned, otherwise True is returned.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#stopmessagelivelocation
     */
    stopMessageLiveLocation(other?: Other<"stopMessageLiveLocation", "chat_id" | "message_id" | "inline_message_id">, signal?: AbortSignal): Promise<true | (Update.Edited & Message.CommonMessage & {
        location: import("@grammyjs/types/message.js").Location;
    })>;
    /**
     * Context-aware alias for `api.sendPaidMedia`. Use this method to send paid media. On success, the sent Message is returned.
     *
     * @param star_count The number of Telegram Stars that must be paid to buy access to the media
     * @param media An array describing the media to be sent; up to 10 items
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendpaidmedia
     */
    sendPaidMedia(star_count: number, media: InputPaidMedia[], other?: Other<"sendPaidMedia", "chat_id" | "star_count" | "media">, signal?: AbortSignal): Promise<Message.PaidMediaMessage>;
    /**
     * Context-aware alias for `api.sendVenue`. Use this method to send information about a venue. On success, the sent Message is returned.
     *
     * @param latitude Latitude of the venue
     * @param longitude Longitude of the venue
     * @param title Name of the venue
     * @param address Address of the venue
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendvenue
     */
    replyWithVenue(latitude: number, longitude: number, title: string, address: string, other?: Other<"sendVenue", "chat_id" | "latitude" | "longitude" | "title" | "address">, signal?: AbortSignal): Promise<Message.VenueMessage>;
    /**
     * Context-aware alias for `api.sendContact`. Use this method to send phone contacts. On success, the sent Message is returned.
     *
     * @param phone_number Contact's phone number
     * @param first_name Contact's first name
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendcontact
     */
    replyWithContact(phone_number: string, first_name: string, other?: Other<"sendContact", "chat_id" | "phone_number" | "first_name">, signal?: AbortSignal): Promise<Message.ContactMessage>;
    /**
     * Context-aware alias for `api.sendPoll`. Use this method to send a native poll. On success, the sent Message is returned.
     *
     * @param question Poll question, 1-300 characters
     * @param options A list of answer options, 2-12 strings 1-100 characters each
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendpoll
     */
    replyWithPoll(question: string, options: (string | InputPollOption)[], other?: Other<"sendPoll", "chat_id" | "question" | "options">, signal?: AbortSignal): Promise<Message.PollMessage>;
    /**
     * Context-aware alias for `api.sendChecklist`. Use this method to send a checklist on behalf of a connected business account. On success, the sent Message is returned.
     *
     * @param checklist An object for the checklist to send
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendchecklist
     */
    replyWithChecklist(checklist: InputChecklist, other?: Other<"sendChecklist", "business_connection_id" | "chat_id" | "checklist">, signal?: AbortSignal): Promise<Message.ChecklistMessage>;
    /**
     * Context-aware alias for `api.editMessageChecklist`. Use this method to edit a checklist on behalf of a connected business account. On success, the edited Message is returned.
     *
     * @param checklist An object for the new checklist
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#editmessagechecklist
     */
    editMessageChecklist(checklist: InputChecklist, other?: Other<"editMessageChecklist", "business_connection_id" | "chat_id" | "messaage_id" | "checklist">, signal?: AbortSignal): Promise<Message.ChecklistMessage>;
    /**
     * Context-aware alias for `api.sendDice`. Use this method to send an animated emoji that will display a random value. On success, the sent Message is returned.
     *
     * @param emoji Emoji on which the dice throw animation is based. Currently, must be one of “🎲”, “🎯”, “🏀”, “⚽”, “🎳”, or “🎰”. Dice can have values 1-6 for “🎲”, “🎯” and “🎳”, values 1-5 for “🏀” and “⚽”, and values 1-64 for “🎰”. Defaults to “🎲”
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#senddice
     */
    replyWithDice(emoji: (string & Record<never, never>) | "🎲" | "🎯" | "🏀" | "⚽" | "🎳" | "🎰", other?: Other<"sendDice", "chat_id" | "emoji">, signal?: AbortSignal): Promise<Message.DiceMessage>;
    /**
     * Context-aware alias for `api.sendChatAction`. Use this method when you need to tell the user that something is happening on the bot's side. The status is set for 5 seconds or less (when a message arrives from your bot, Telegram clients clear its typing status). Returns True on success.
     *
     * Example: The ImageBot needs some time to process a request and upload the image. Instead of sending a text message along the lines of “Retrieving image, please wait…”, the bot may use sendChatAction with action = upload_photo. The user will see a “sending photo” status for the bot.
     *
     * We only recommend using this method when a response from the bot will take a noticeable amount of time to arrive.
     *
     * @param action Type of action to broadcast. Choose one, depending on what the user is about to receive: typing for text messages, upload_photo for photos, record_video or upload_video for videos, record_voice or upload_voice for voice notes, upload_document for general files, choose_sticker for stickers, find_location for location data, record_video_note or upload_video_note for video notes.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendchataction
     */
    replyWithChatAction(action: "typing" | "upload_photo" | "record_video" | "upload_video" | "record_voice" | "upload_voice" | "upload_document" | "choose_sticker" | "find_location" | "record_video_note" | "upload_video_note", other?: Other<"sendChatAction", "chat_id" | "action">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setMessageReaction`. Use this method to change the chosen reactions on a message. Service messages of some types can't be reacted to. Automatically forwarded messages from a channel to its discussion group have the same available reactions as messages in the channel. Bots can't use paid reactions. Returns True on success.
     *
     * @param reaction A list of reaction types to set on the message. Currently, as non-premium users, bots can set up to one reaction per message. A custom emoji reaction can be used if it is either already present on the message or explicitly allowed by chat administrators. Paid reactions can't be used by bots.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setmessagereaction
     */
    react(reaction: MaybeArray<ReactionTypeEmoji["emoji"] | ReactionType>, other?: Other<"setMessageReaction", "chat_id" | "message_id" | "reaction">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.getUserProfilePhotos`. Use this method to get a list of profile pictures for a user. Returns a UserProfilePhotos object.
     *
     * @param user_id Unique identifier of the target user
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getuserprofilephotos
     */
    getUserProfilePhotos(other?: Other<"getUserProfilePhotos", "user_id">, signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").UserProfilePhotos>;
    /**
     * Context-aware alias for `api.serUserEmojiStatus`. Changes the emoji status for a given user that previously allowed the bot to manage their emoji status via the Mini App method requestEmojiStatusAccess. Returns True on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setuseremojistatus
     */
    setUserEmojiStatus(other?: Other<"setUserEmojiStatus", "user_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.getUserChatBoosts`. Use this method to get the list of boosts added to a chat by a user. Requires administrator rights in the chat. Returns a UserChatBoosts object.
     *
     * @param chat_id Unique identifier for the chat or username of the channel (in the format @channelusername)
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getuserchatboosts
     */
    getUserChatBoosts(chat_id?: number | string, signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").UserChatBoosts>;
    /**
     * Context-aware alias for `api.getUserGifts`. Returns the gifts owned and hosted by a user. Returns OwnedGifts on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getusergifts
     */
    getUserGifts(other?: Other<"getUserGifts", "user_id">, signal?: AbortSignal): Promise<import("@grammyjs/types/payment.js").OwnedGifts>;
    /**
     * Context-aware alias for `api.getChatGifts`. Returns the gifts owned by a chat. Returns OwnedGifts on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getchatgifts
     */
    getChatGifts(other?: Other<"getChatGifts", "chat_id">, signal?: AbortSignal): Promise<import("@grammyjs/types/payment.js").OwnedGifts>;
    /**
     * Context-aware alias for `api.getBusinessConnection`. Use this method to get information about the connection of the bot with a business account. Returns a BusinessConnection object on success.
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getbusinessconnection
     */
    getBusinessConnection(signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").BusinessConnection>;
    /**
     * Context-aware alias for `api.getFile`. Use this method to get basic info about a file and prepare it for downloading. For the moment, bots can download files of up to 20MB in size. On success, a File object is returned. The file can then be downloaded via the link https://api.telegram.org/file/bot<token>/<file_path>, where <file_path> is taken from the response. It is guaranteed that the link will be valid for at least 1 hour. When the link expires, a new one can be requested by calling getFile again.
     *
     * Note: This function may not preserve the original file name and MIME type. You should save the file's MIME type and name (if available) when the File object is received.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getfile
     */
    getFile(signal?: AbortSignal): Promise<import("@grammyjs/types/message.js").File>;
    /** @deprecated Use `banAuthor` instead. */
    kickAuthor(...args: Parameters<Context["banAuthor"]>): Promise<true>;
    /**
     * Context-aware alias for `api.banChatMember`. Use this method to ban a user in a group, a supergroup or a channel. In the case of supergroups and channels, the user will not be able to return to the chat on their own using invite links, etc., unless unbanned first. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#banchatmember
     */
    banAuthor(other?: Other<"banChatMember", "chat_id" | "user_id">, signal?: AbortSignal): Promise<true>;
    /** @deprecated Use `banChatMember` instead. */
    kickChatMember(...args: Parameters<Context["banChatMember"]>): Promise<true>;
    /**
     * Context-aware alias for `api.banChatMember`. Use this method to ban a user in a group, a supergroup or a channel. In the case of supergroups and channels, the user will not be able to return to the chat on their own using invite links, etc., unless unbanned first. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
     *
     * @param user_id Unique identifier of the target user
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#banchatmember
     */
    banChatMember(user_id: number, other?: Other<"banChatMember", "chat_id" | "user_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.unbanChatMember`. Use this method to unban a previously banned user in a supergroup or channel. The user will not return to the group or channel automatically, but will be able to join via link, etc. The bot must be an administrator for this to work. By default, this method guarantees that after the call the user is not a member of the chat, but will be able to join it. So if the user is a member of the chat they will also be removed from the chat. If you don't want this, use the parameter only_if_banned. Returns True on success.
     *
     * @param user_id Unique identifier of the target user
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#unbanchatmember
     */
    unbanChatMember(user_id: number, other?: Other<"unbanChatMember", "chat_id" | "user_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.restrictChatMember`. Use this method to restrict a user in a supergroup. The bot must be an administrator in the supergroup for this to work and must have the appropriate administrator rights. Pass True for all permissions to lift restrictions from a user. Returns True on success.
     *
     * @param permissions An object for new user permissions
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#restrictchatmember
     */
    restrictAuthor(permissions: ChatPermissions, other?: Other<"restrictChatMember", "chat_id" | "user_id" | "permissions">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.restrictChatMember`. Use this method to restrict a user in a supergroup. The bot must be an administrator in the supergroup for this to work and must have the appropriate administrator rights. Pass True for all permissions to lift restrictions from a user. Returns True on success.
     *
     * @param user_id Unique identifier of the target user
     * @param permissions An object for new user permissions
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#restrictchatmember
     */
    restrictChatMember(user_id: number, permissions: ChatPermissions, other?: Other<"restrictChatMember", "chat_id" | "user_id" | "permissions">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.promoteChatMember`. Use this method to promote or demote a user in a supergroup or a channel. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Pass False for all boolean parameters to demote a user. Returns True on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#promotechatmember
     */
    promoteAuthor(other?: Other<"promoteChatMember", "chat_id" | "user_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.promoteChatMember`. Use this method to promote or demote a user in a supergroup or a channel. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Pass False for all boolean parameters to demote a user. Returns True on success.
     *
     * @param user_id Unique identifier of the target user
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#promotechatmember
     */
    promoteChatMember(user_id: number, other?: Other<"promoteChatMember", "chat_id" | "user_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setChatAdministratorCustomTitle`. Use this method to set a custom title for an administrator in a supergroup promoted by the bot. Returns True on success.
     *
     * @param custom_title New custom title for the administrator; 0-16 characters, emoji are not allowed
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setchatadministratorcustomtitle
     */
    setChatAdministratorAuthorCustomTitle(custom_title: string, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setChatAdministratorCustomTitle`. Use this method to set a custom title for an administrator in a supergroup promoted by the bot. Returns True on success.
     *
     * @param user_id Unique identifier of the target user
     * @param custom_title New custom title for the administrator; 0-16 characters, emoji are not allowed
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setchatadministratorcustomtitle
     */
    setChatAdministratorCustomTitle(user_id: number, custom_title: string, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.banChatSenderChat`. Use this method to ban a channel chat in a supergroup or a channel. Until the chat is unbanned, the owner of the banned chat won't be able to send messages on behalf of any of their channels. The bot must be an administrator in the supergroup or channel for this to work and must have the appropriate administrator rights. Returns True on success.
     *
     * @param sender_chat_id Unique identifier of the target sender chat
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#banchatsenderchat
     */
    banChatSenderChat(sender_chat_id: number, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.unbanChatSenderChat`. Use this method to unban a previously banned channel chat in a supergroup or channel. The bot must be an administrator for this to work and must have the appropriate administrator rights. Returns True on success.
     *
     * @param sender_chat_id Unique identifier of the target sender chat
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#unbanchatsenderchat
     */
    unbanChatSenderChat(sender_chat_id: number, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setChatPermissions`. Use this method to set default chat permissions for all members. The bot must be an administrator in the group or a supergroup for this to work and must have the can_restrict_members administrator rights. Returns True on success.
     *
     * @param permissions New default chat permissions
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setchatpermissions
     */
    setChatPermissions(permissions: ChatPermissions, other?: Other<"setChatPermissions", "chat_id" | "permissions">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.exportChatInviteLink`. Use this method to generate a new primary invite link for a chat; any previously generated primary link is revoked. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns the new invite link as String on success.
     *
     * Note: Each administrator in a chat generates their own invite links. Bots can't use invite links generated by other administrators. If you want your bot to work with invite links, it will need to generate its own link using exportChatInviteLink or by calling the getChat method. If your bot needs to generate a new primary invite link replacing its previous one, use exportChatInviteLink again.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#exportchatinvitelink
     */
    exportChatInviteLink(signal?: AbortSignal): Promise<string>;
    /**
     * Context-aware alias for `api.createChatInviteLink`. Use this method to create an additional invite link for a chat. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. The link can be revoked using the method revokeChatInviteLink. Returns the new invite link as ChatInviteLink object.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#createchatinvitelink
     */
    createChatInviteLink(other?: Other<"createChatInviteLink", "chat_id">, signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").ChatInviteLink>;
    /**
     * Context-aware alias for `api.editChatInviteLink`. Use this method to edit a non-primary invite link created by the bot. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns the edited invite link as a ChatInviteLink object.
     *
     * @param invite_link The invite link to edit
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#editchatinvitelink
     */
    editChatInviteLink(invite_link: string, other?: Other<"editChatInviteLink", "chat_id" | "invite_link">, signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").ChatInviteLink>;
    /**
     * Context-aware alias for `api.createChatSubscriptionInviteLink`. Use this method to create a subscription invite link for a channel chat. The bot must have the can_invite_users administrator rights. The link can be edited using the method editChatSubscriptionInviteLink or revoked using the method revokeChatInviteLink. Returns the new invite link as a ChatInviteLink object.
     *
     * @param subscription_period The number of seconds the subscription will be active for before the next payment. Currently, it must always be 2592000 (30 days).
     * @param subscription_price The amount of Telegram Stars a user must pay initially and after each subsequent subscription period to be a member of the chat; 1-2500
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#createchatsubscriptioninvitelink
     */
    createChatSubscriptionInviteLink(subscription_period: number, subscription_price: number, other?: Other<"createChatSubscriptionInviteLink", "chat_id" | "subscription_period" | "subscription_price">, signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").ChatInviteLink>;
    /**
     * Context-aware alias for `api.editChatSubscriptionInviteLink`. Use this method to edit a subscription invite link created by the bot. The bot must have the can_invite_users administrator rights. Returns the edited invite link as a ChatInviteLink object.
     *
     * @param invite_link The invite link to edit
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#editchatsubscriptioninvitelink
     */
    editChatSubscriptionInviteLink(invite_link: string, other?: Other<"editChatSubscriptionInviteLink", "chat_id" | "invite_link">, signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").ChatInviteLink>;
    /**
     * Context-aware alias for `api.revokeChatInviteLink`. Use this method to revoke an invite link created by the bot. If the primary link is revoked, a new link is automatically generated. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns the revoked invite link as ChatInviteLink object.
     *
     * @param invite_link The invite link to revoke
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#revokechatinvitelink
     */
    revokeChatInviteLink(invite_link: string, signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").ChatInviteLink>;
    /**
     * Context-aware alias for `api.approveChatJoinRequest`. Use this method to approve a chat join request. The bot must be an administrator in the chat for this to work and must have the can_invite_users administrator right. Returns True on success.
     *
     * @param user_id Unique identifier of the target user
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#approvechatjoinrequest
     */
    approveChatJoinRequest(user_id: number, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.declineChatJoinRequest`. Use this method to decline a chat join request. The bot must be an administrator in the chat for this to work and must have the can_invite_users administrator right. Returns True on success.
     *
     * @param user_id Unique identifier of the target user
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#declinechatjoinrequest
     */
    declineChatJoinRequest(user_id: number, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.approveSuggestedPost`. Use this method to approve a suggested post in a direct messages chat. The bot must have the 'can_post_messages' administrator right in the corresponding channel chat.  Returns True on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#approvesuggestedpost
     */
    approveSuggestedPost(other?: Other<"approveSuggestedPost", "chat_id" | "message_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.declineSuggestedPost`. Use this method to decline a suggested post in a direct messages chat. The bot must have the 'can_manage_direct_messages' administrator right in the corresponding channel chat. Returns True on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#declinesuggestedpost
     */
    declineSuggestedPost(other?: Other<"declineSuggestedPost", "chat_id" | "message_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setChatPhoto`. Use this method to set a new profile photo for the chat. Photos can't be changed for private chats. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
     *
     * @param photo New chat photo, uploaded using multipart/form-data
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setchatphoto
     */
    setChatPhoto(photo: InputFile, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.deleteChatPhoto`. Use this method to delete a chat photo. Photos can't be changed for private chats. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#deletechatphoto
     */
    deleteChatPhoto(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setChatTitle`. Use this method to change the title of a chat. Titles can't be changed for private chats. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
     *
     * @param title New chat title, 1-255 characters
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setchattitle
     */
    setChatTitle(title: string, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setChatDescription`. Use this method to change the description of a group, a supergroup or a channel. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
     *
     * @param description New chat description, 0-255 characters
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setchatdescription
     */
    setChatDescription(description: string | undefined, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.pinChatMessage`. Use this method to add a message to the list of pinned messages in a chat. In private chats and channel direct messages chats, all non-service messages can be pinned. Conversely, the bot must be an administrator with the 'can_pin_messages' right or the 'can_edit_messages' right to pin messages in groups and channels respectively. Returns True on success.
     *
     * @param message_id Identifier of a message to pin
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#pinchatmessage
     */
    pinChatMessage(message_id: number, other?: Other<"pinChatMessage", "chat_id" | "message_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.unpinChatMessage`. Use this method to remove a message from the list of pinned messages in a chat. In private chats and channel direct messages chats, all messages can be unpinned. Conversely, the bot must be an administrator with the 'can_pin_messages' right or the 'can_edit_messages' right to unpin messages in groups and channels respectively. Returns True on success.
     *
     * @param message_id Identifier of a message to unpin. If not specified, the most recent pinned message (by sending date) will be unpinned.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#unpinchatmessage
     */
    unpinChatMessage(message_id?: number, other?: Other<"unpinChatMessage", "chat_id" | "message_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.unpinAllChatMessages`. Use this method to clear the list of pinned messages in a chat. In private chats and channel direct messages chats, no additional rights are required to unpin all pinned messages. Conversely, the bot must be an administrator with the 'can_pin_messages' right or the 'can_edit_messages' right to unpin all pinned messages in groups and channels respectively. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#unpinallchatmessages
     */
    unpinAllChatMessages(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.leaveChat`. Use this method for your bot to leave a group, supergroup or channel. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#leavechat
     */
    leaveChat(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.getChat`. Use this method to get up to date information about the chat (current name of the user for one-on-one conversations, current username of a user, group or channel, etc.). Returns a Chat object on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getchat
     */
    getChat(signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").ChatFullInfo>;
    /**
     * Context-aware alias for `api.getChatAdministrators`. Use this method to get a list of administrators in a chat, which aren't bots. Returns an Array of ChatMember objects.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getchatadministrators
     */
    getChatAdministrators(signal?: AbortSignal): Promise<(import("@grammyjs/types/manage.js").ChatMemberOwner | import("@grammyjs/types/manage.js").ChatMemberAdministrator)[]>;
    /** @deprecated Use `getChatMemberCount` instead. */
    getChatMembersCount(...args: Parameters<Context["getChatMemberCount"]>): Promise<number>;
    /**
     * Context-aware alias for `api.getChatMemberCount`. Use this method to get the number of members in a chat. Returns Int on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getchatmembercount
     */
    getChatMemberCount(signal?: AbortSignal): Promise<number>;
    /**
     * Context-aware alias for `api.getChatMember`. Use this method to get information about a member of a chat. The method is guaranteed to work only if the bot is an administrator in the chat. Returns a ChatMember object on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getchatmember
     */
    getAuthor(signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").ChatMember>;
    /**
     * Context-aware alias for `api.getChatMember`. Use this method to get information about a member of a chat. The method is guaranteed to work only if the bot is an administrator in the chat. Returns a ChatMember object on success.
     *
     * @param user_id Unique identifier of the target user
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getchatmember
     */
    getChatMember(user_id: number, signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").ChatMember>;
    /**
     * Context-aware alias for `api.setChatStickerSet`. Use this method to set a new group sticker set for a supergroup. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Use the field can_set_sticker_set ly returned in getChat requests to check if the bot can use this method. Returns True on success.
     *
     * @param sticker_set_name Name of the sticker set to be set as the group sticker set
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setchatstickerset
     */
    setChatStickerSet(sticker_set_name: string, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.deleteChatStickerSet`. Use this method to delete a group sticker set from a supergroup. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Use the field can_set_sticker_set ly returned in getChat requests to check if the bot can use this method. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#deletechatstickerset
     */
    deleteChatStickerSet(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.createForumTopic`. Use this method to create a topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights. Returns information about the created topic as a ForumTopic object.
     *
     * @param name Topic name, 1-128 characters
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#createforumtopic
     */
    createForumTopic(name: string, other?: Other<"createForumTopic", "chat_id" | "name">, signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").ForumTopic>;
    /**
     * Context-aware alias for `api.editForumTopic`. Use this method to edit name and icon of a topic in a forum supergroup chat or a private chat with a user. In the case of a supergroup chat the bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights, unless it is the creator of the topic. Returns True on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#editforumtopic
     */
    editForumTopic(other?: Other<"editForumTopic", "chat_id" | "message_thread_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.closeForumTopic`. Use this method to close an open topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights, unless it is the creator of the topic. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#closeforumtopic
     */
    closeForumTopic(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.reopenForumTopic`. Use this method to reopen a closed topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights, unless it is the creator of the topic. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#reopenforumtopic
     */
    reopenForumTopic(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.deleteForumTopic`. Use this method to delete a forum topic along with all its messages in a forum supergroup chat or a private chat with a user. In the case of a supergroup chat the bot must be an administrator in the chat for this to work and must have the can_delete_messages administrator rights. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#deleteforumtopic
     */
    deleteForumTopic(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.unpinAllForumTopicMessages`. Use this method to clear the list of pinned messages in a forum topic in a forum supergroup chat or a private chat with a user. In the case of a supergroup chat the bot must be an administrator in the chat for this to work and must have the can_pin_messages administrator right in the supergroup. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#unpinallforumtopicmessages
     */
    unpinAllForumTopicMessages(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.editGeneralForumTopic`. Use this method to edit the name of the 'General' topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights. Returns True on success.
     *
     * @param name New topic name, 1-128 characters
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#editgeneralforumtopic
     */
    editGeneralForumTopic(name: string, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.closeGeneralForumTopic`. Use this method to close an open 'General' topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#closegeneralforumtopic
     */
    closeGeneralForumTopic(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.reopenGeneralForumTopic`. Use this method to reopen a closed 'General' topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights. The topic will be automatically unhidden if it was hidden. Returns True on success.     *
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#reopengeneralforumtopic
     */
    reopenGeneralForumTopic(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.hideGeneralForumTopic`. Use this method to hide the 'General' topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights. The topic will be automatically closed if it was open. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#hidegeneralforumtopic
     */
    hideGeneralForumTopic(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.unhideGeneralForumTopic`. Use this method to unhide the 'General' topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#unhidegeneralforumtopic
     */
    unhideGeneralForumTopic(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.unpinAllGeneralForumTopicMessages`. Use this method to clear the list of pinned messages in a General forum topic. The bot must be an administrator in the chat for this to work and must have the can_pin_messages administrator right in the supergroup. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#unpinallgeneralforumtopicmessages
     */
    unpinAllGeneralForumTopicMessages(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.answerCallbackQuery`. Use this method to send answers to callback queries sent from inline keyboards. The answer will be displayed to the user as a notification at the top of the chat screen or as an alert. On success, True is returned.
     *
     * Alternatively, the user can be redirected to the specified Game URL. For this option to work, you must first create a game for your bot via @BotFather and accept the terms. Otherwise, you may use links like t.me/your_bot?start=XXXX that open your bot with a parameter.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#answercallbackquery
     */
    answerCallbackQuery(other?: string | Other<"answerCallbackQuery", "callback_query_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setChatMenuButton`. Use this method to change the bot's menu button in a private chat, or the default menu button. Returns True on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setchatmenubutton
     */
    setChatMenuButton(other?: Other<"setChatMenuButton">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.getChatMenuButton`. Use this method to get the current value of the bot's menu button in a private chat, or the default menu button. Returns MenuButton on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getchatmenubutton
     */
    getChatMenuButton(other?: Other<"getChatMenuButton">, signal?: AbortSignal): Promise<import("@grammyjs/types/settings.js").MenuButton>;
    /**
     * Context-aware alias for `api.setMyDefaultAdministratorRights`. Use this method to the change the default administrator rights requested by the bot when it's added as an administrator to groups or channels. These rights will be suggested to users, but they are are free to modify the list before adding the bot. Returns True on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setmydefaultadministratorrights
     */
    setMyDefaultAdministratorRights(other?: Other<"setMyDefaultAdministratorRights">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.getMyDefaultAdministratorRights`. Use this method to get the current default administrator rights of the bot. Returns ChatAdministratorRights on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     */
    getMyDefaultAdministratorRights(other?: Other<"getMyDefaultAdministratorRights">, signal?: AbortSignal): Promise<import("@grammyjs/types/manage.js").ChatAdministratorRights>;
    /**
     * Context-aware alias for `api.editMessageText`. Use this method to edit text and game messages. On success, if the edited message is not an inline message, the edited Message is returned, otherwise True is returned. Note that business messages that were not sent by the bot and do not contain an inline keyboard can only be edited within 48 hours from the time they were sent.
     *
     * @param text New text of the message, 1-4096 characters after entities parsing
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#editmessagetext
     */
    editMessageText(text: string, other?: Other<"editMessageText", "chat_id" | "message_id" | "inline_message_id" | "text">, signal?: AbortSignal): Promise<true | (Update.Edited & Message.CommonMessage & {
        text: string;
    } & Partial<{
        entities: MessageEntity[];
    }>)>;
    /**
     * Context-aware alias for `api.editMessageCaption`. Use this method to edit captions of messages. On success, if the edited message is not an inline message, the edited Message is returned, otherwise True is returned. Note that business messages that were not sent by the bot and do not contain an inline keyboard can only be edited within 48 hours from the time they were sent.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#editmessagecaption
     */
    editMessageCaption(other?: Other<"editMessageCaption", "chat_id" | "message_id" | "inline_message_id">, signal?: AbortSignal): Promise<true | (Update.Edited & Message.CaptionableMessage)>;
    /**
     * Context-aware alias for `api.editMessageMedia`. Use this method to edit animation, audio, document, photo, or video messages, or to add media to text messages. If a message is part of a message album, then it can be edited only to an audio for audio albums, only to a document for document albums and to a photo or a video otherwise. When an inline message is edited, a new file can't be uploaded; use a previously uploaded file via its file_id or specify a URL. On success, if the edited message is not an inline message, the edited Message is returned, otherwise True is returned. Note that business messages that were not sent by the bot and do not contain an inline keyboard can only be edited within 48 hours from the time they were sent.
     *
     * @param media An object for a new media content of the message
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#editmessagemedia
     */
    editMessageMedia(media: InputMedia, other?: Other<"editMessageMedia", "chat_id" | "message_id" | "inline_message_id" | "media">, signal?: AbortSignal): Promise<true | (Update.Edited & Message)>;
    /**
     * Context-aware alias for `api.editMessageReplyMarkup`. Use this method to edit only the reply markup of messages. On success, if the edited message is not an inline message, the edited Message is returned, otherwise True is returned. Note that business messages that were not sent by the bot and do not contain an inline keyboard can only be edited within 48 hours from the time they were sent.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#editmessagereplymarkup
     */
    editMessageReplyMarkup(other?: Other<"editMessageReplyMarkup", "chat_id" | "message_id" | "inline_message_id">, signal?: AbortSignal): Promise<true | (Update.Edited & Message)>;
    /**
     * Context-aware alias for `api.stopPoll`. Use this method to stop a poll which was sent by the bot. On success, the stopped Poll is returned.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#stoppoll
     */
    stopPoll(other?: Other<"stopPoll", "chat_id" | "message_id">, signal?: AbortSignal): Promise<import("@grammyjs/types/message.js").Poll>;
    /**
     * Context-aware alias for `api.deleteMessage`. Use this method to delete a message, including service messages, with the following limitations:
     * - A message can only be deleted if it was sent less than 48 hours ago.
     * - A dice message in a private chat can only be deleted if it was sent more than 24 hours ago.
     * - Bots can delete outgoing messages in private chats, groups, and supergroups.
     * - Bots can delete incoming messages in private chats.
     * - Bots granted can_post_messages permissions can delete outgoing messages in channels.
     * - If the bot is an administrator of a group, it can delete any message there.
     * - If the bot has can_delete_messages administrator right in a supergroup or a channel, it can delete any message there.
     * - If the bot has can_manage_direct_messages administrator right in a channel, it can delete any message in the corresponding direct messages chat.
     * Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#deletemessage
     */
    deleteMessage(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.deleteMessages`. Use this method to delete multiple messages simultaneously. Returns True on success.
     *
     * @param chat_id Unique identifier for the target chat or username of the target channel (in the format @channelusername)
     * @param message_ids A list of 1-100 identifiers of messages to delete. See deleteMessage for limitations on which messages can be deleted
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#deletemessages
     */
    deleteMessages(message_ids: number[], signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.deleteBusinessMessages`. Delete messages on behalf of a business account. Requires the can_delete_outgoing_messages business bot right to delete messages sent by the bot itself, or the can_delete_all_messages business bot right to delete any message. Returns True on success.
     *
     * @param message_ids A list of 1-100 identifiers of messages to delete. All messages must be from the same chat. See deleteMessage for limitations on which messages can be deleted
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#deletebusinessmessages
     */
    deleteBusinessMessages(message_ids: number[], signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setBusinessAccountName`. Changes the first and last name of a managed business account. Requires the can_change_name business bot right. Returns True on success.
     *
     * @param first_name The new value of the first name for the business account; 1-64 characters
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setbusinessaccountname
     */
    setBusinessAccountName(first_name: string, other: Other<"setBusinessAccountName", "business_connection_id" | "first_name">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setBusinessAccountUsername`. Changes the username of a managed business account. Requires the can_change_username business bot right. Returns True on success.
     *
     * @param username The new value of the username for the business account; 0-32 characters
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setbusinessaccountusername
     */
    setBusinessAccountUsername(username: string, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setBusinessAccountBio`. Changes the bio of a managed business account. Requires the can_change_bio business bot right. Returns True on success.
     *
     * @param bio The new value of the bio for the business account; 0-140 characters
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setbusinessaccountbio
     */
    setBusinessAccountBio(bio: string, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setBusinessAccountProfilePhoto`. CsetBusinessAccountProfilePhotohanges the profile photo of a managed business account. Requires the can_edit_profile_photo business bot right. Returns True on success.
     *
     * @param photo The new profile photo to set
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setbusinessaccountprofilephoto
     */
    setBusinessAccountProfilePhoto(photo: InputProfilePhoto, other: Other<"setBusinessAccountProfilePhoto", "business_connection_id" | "photo">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.removeBusinessAccountProfilePhoto`. Removes the current profile photo of a managed business account. Requires the can_edit_profile_photo business bot right. Returns True on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#removebusinessaccountprofilephoto
     */
    removeBusinessAccountProfilePhoto(other: Other<"removeBusinessAccountProfilePhoto", "business_connection_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setBusinessAccountGiftSettings`. Changes the privacy settings pertaining to incoming gifts in a managed business account. Requires the can_change_gift_settings business bot right. Returns True on success.
     *
     * @param show_gift_button Pass True, if a button for sending a gift to the user or by the business account must always be shown in the input field
     * @param accepted_gift_types Types of gifts accepted by the business account
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setbusinessaccountgiftsettings
     */
    setBusinessAccountGiftSettings(show_gift_button: boolean, accepted_gift_types: AcceptedGiftTypes, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.getBusinessAccountStarBalance`. Returns the amount of Telegram Stars owned by a managed business account. Requires the can_view_gifts_and_stars business bot right. Returns StarAmount on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getbusinessaccountstarbalance
     */
    getBusinessAccountStarBalance(signal?: AbortSignal): Promise<import("@grammyjs/types/payment.js").StarAmount>;
    /**
     * Context-aware alias for `api.transferBusinessAccountStars`. Transfers Telegram Stars from the business account balance to the bot's balance. Requires the can_transfer_stars business bot right. Returns True on success.
     *
     * @param star_count Number of Telegram Stars to transfer; 1-10000
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#transferbusinessaccountstars
     */
    transferBusinessAccountStars(star_count: number, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.getBusinessAccountGifts`. Returns the gifts received and owned by a managed business account. Requires the can_view_gifts_and_stars business bot right. Returns OwnedGifts on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getbusinessaccountgifts
     */
    getBusinessAccountGifts(other: Other<"getBusinessAccountGifts", "business_connection_id">, signal?: AbortSignal): Promise<import("@grammyjs/types/payment.js").OwnedGifts>;
    /**
     * Context-aware alias for `api.convertGiftToStars`. Converts a given regular gift to Telegram Stars. Requires the can_convert_gifts_to_stars business bot right. Returns True on success.
     *
     * @param owned_gift_id Unique identifier of the regular gift that should be converted to Telegram Stars
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#convertgifttostars
     */
    convertGiftToStars(owned_gift_id: string, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.upgradeGift`. Upgrades a given regular gift to a unique gift. Requires the can_transfer_and_upgrade_gifts business bot right. Additionally requires the can_transfer_stars business bot right if the upgrade is paid. Returns True on success.
     *
     * @param owned_gift_id Unique identifier of the regular gift that should be upgraded to a unique one
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#upgradegift
     */
    upgradeGift(owned_gift_id: string, other: Other<"getBusinessAccountGifts", "business_connection_id" | "owned_gift_id">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.transferGift`. Transfers an owned unique gift to another user. Requires the can_transfer_and_upgrade_gifts business bot right. Requires can_transfer_stars business bot right if the transfer is paid. Returns True on success.
     *
     * @param owned_gift_id Unique identifier of the regular gift that should be transferred
     * @param new_owner_chat_id Unique identifier of the chat which will own the gift. The chat must be active in the last 24 hours.
     * @param star_count The amount of Telegram Stars that will be paid for the transfer from the business account balance. If positive, then the can_transfer_stars business bot right is required.
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#transfergift
     */
    transferGift(owned_gift_id: string, new_owner_chat_id: number, star_count: number, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.postStory`. Posts a story on behalf of a managed business account. Requires the can_manage_stories business bot right. Returns Story on success.
     *
     * @param content Content of the story
     * @param active_period Period after which the story is moved to the archive, in seconds; must be one of 6 * 3600, 12 * 3600, 86400, or 2 * 86400
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#poststory
     */
    postStory(content: InputStoryContent, active_period: number, other: Other<"postStory", "business_connection_id" | "content" | "active_period">, signal?: AbortSignal): Promise<import("@grammyjs/types/message.js").Story>;
    /**
     * Context-aware alias for `api.repostStory`. Reposts a story on behalf of a business account from another business account. Both business accounts must be managed by the same bot, and the story on the source account must have been posted (or reposted) by the bot. Requires the can_manage_stories business bot right for both business accounts. Returns Story on success.
     *
     * @param active_period Period after which the story is moved to the archive, in seconds; must be one of 6 * 3600, 12 * 3600, 86400, or 2 * 86400
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#repoststory
     */
    repostStory(active_period: number, other: Other<"repostStory", "business_connection_id" | "from_chat_id" | "from_story_id" | "active_period">, signal?: AbortSignal): Promise<import("@grammyjs/types/message.js").Story>;
    /**
     * Context-aware alias for `api.editStory`. Edits a story previously posted by the bot on behalf of a managed business account. Requires the can_manage_stories business bot right. Returns Story on success.
     *
     * @param story_id Unique identifier of the story to edit
     * @param content Content of the story
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#editstory
     */
    editStory(story_id: number, content: InputStoryContent, other: Other<"editStory", "business_connection_id" | "story_id" | "content">, signal?: AbortSignal): Promise<import("@grammyjs/types/message.js").Story>;
    /**
     * Context-aware alias for `api.deleteStory`. Deletes a story previously posted by the bot on behalf of a managed business account. Requires the can_manage_stories business bot right. Returns True on success.
     *
     * @param story_id Unique identifier of the story to delete
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#deletestory
     */
    deleteStory(story_id: number, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.sendSticker`. Use this method to send static .WEBP, animated .TGS, or video .WEBM stickers. On success, the sent Message is returned.
     *
     * @param sticker Sticker to send. Pass a file_id as String to send a file that exists on the Telegram servers (recommended), pass an HTTP URL as a String for Telegram to get a .WEBP sticker from the Internet, or upload a new .WEBP, .TGS, or .WEBM sticker using multipart/form-data. Video and animated stickers can't be sent via an HTTP URL.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendsticker
     */
    replyWithSticker(sticker: InputFile | string, other?: Other<"sendSticker", "chat_id" | "sticker">, signal?: AbortSignal): Promise<Message.StickerMessage>;
    /**
     * Use this method to get information about custom emoji stickers by their identifiers. Returns an Array of Sticker objects.
     *
     * @param custom_emoji_ids A list of custom emoji identifiers
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#getcustomemojistickers
     */
    getCustomEmojiStickers(signal?: AbortSignal): Promise<import("@grammyjs/types/message.js").Sticker[]>;
    /**
     * Context-aware alias for `api.sendGift`. Sends a gift to the given user. The gift can't be converted to Telegram Stars by the receiver. Returns True on success.
     *
     * @param gift_id Identifier of the gift
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendgift
     */
    replyWithGift(gift_id: string, other?: Other<"sendGift", "user_id" | "chat_id" | "gift_id">, signal?: AbortSignal): Promise<import("@grammyjs/types/payment.js").Gifts>;
    /**
     * Context-aware alias for `api.giftPremiumSubscription`. Gifts a Telegram Premium subscription to the given user. Returns True on success.
     *
     * @param month_count Number of months the Telegram Premium subscription will be active for the user; must be one of 3, 6, or 12
     * @param star_count Number of Telegram Stars to pay for the Telegram Premium subscription; must be 1000 for 3 months, 1500 for 6 months, and 2500 for 12 months
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#giftpremiumsubscription
     */
    giftPremiumSubscription(month_count: 3 | 6 | 12, star_count: 1000 | 1500 | 2500, other?: Other<"giftPremiumSubscription", "user_id" | "month_count" | "star_count">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.sendGift`. Sends a gift to the given channel chat. The gift can't be converted to Telegram Stars by the receiver. Returns True on success.
     *
     * @param gift_id Identifier of the gift
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendgift
     */
    replyWithGiftToChannel(gift_id: string, other?: Other<"sendGift", "user_id" | "chat_id" | "gift_id">, signal?: AbortSignal): Promise<import("@grammyjs/types/payment.js").Gifts>;
    /**
     * Context-aware alias for `api.answerInlineQuery`. Use this method to send answers to an inline query. On success, True is returned.
     * No more than 50 results per query are allowed.
     *
     * Example: An inline bot that sends YouTube videos can ask the user to connect the bot to their YouTube account to adapt search results accordingly. To do this, it displays a 'Connect your YouTube account' button above the results, or even before showing any. The user presses the button, switches to a private chat with the bot and, in doing so, passes a start parameter that instructs the bot to return an OAuth link. Once done, the bot can offer a switch_inline button so that the user can easily return to the chat where they wanted to use the bot's inline capabilities.
     *
     * @param results An array of results for the inline query
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#answerinlinequery
     */
    answerInlineQuery(results: readonly InlineQueryResult[], other?: Other<"answerInlineQuery", "inline_query_id" | "results">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.savePreparedInlineMessage`. Stores a message that can be sent by a user of a Mini App. Returns a PreparedInlineMessage object.
     *
     * @param result An object describing the message to be sent
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#savepreparedinlinemessage
     */
    savePreparedInlineMessage(result: InlineQueryResult, other?: Other<"savePreparedInlineMessage", "user_id" | "result">, signal?: AbortSignal): Promise<import("@grammyjs/types/message.js").PreparedInlineMessage>;
    /**
     * Context-aware alias for `api.sendInvoice`. Use this method to send invoices. On success, the sent Message is returned.
     *
     * @param title Product name, 1-32 characters
     * @param description Product description, 1-255 characters
     * @param payload Bot-defined invoice payload, 1-128 bytes. This will not be displayed to the user, use for your internal processes.
     * @param currency Three-letter ISO 4217 currency code, see more on currencies
     * @param prices Price breakdown, a list of components (e.g. product price, tax, discount, delivery cost, delivery tax, bonus, etc.)
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendinvoice
     */
    replyWithInvoice(title: string, description: string, payload: string, currency: string, prices: readonly LabeledPrice[], other?: Other<"sendInvoice", "chat_id" | "title" | "description" | "payload" | "currency" | "prices">, signal?: AbortSignal): Promise<Message.InvoiceMessage>;
    /**
     * Context-aware alias for `api.answerShippingQuery`. If you sent an invoice requesting a shipping address and the parameter is_flexible was specified, the Bot API will send an Update with a shipping_query field to the bot. Use this method to reply to shipping queries. On success, True is returned.
     *
     * @param shipping_query_id Unique identifier for the query to be answered
     * @param ok Pass True if delivery to the specified address is possible and False if there are any problems (for example, if delivery to the specified address is not possible)
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#answershippingquery
     */
    answerShippingQuery(ok: boolean, other?: Other<"answerShippingQuery", "shipping_query_id" | "ok">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.answerPreCheckoutQuery`. Once the user has confirmed their payment and shipping details, the Bot API sends the final confirmation in the form of an Update with the field pre_checkout_query. Use this method to respond to such pre-checkout queries. On success, True is returned. Note: The Bot API must receive an answer within 10 seconds after the pre-checkout query was sent.
     *
     * @param ok Specify True if everything is alright (goods are available, etc.) and the bot is ready to proceed with the order. Use False if there are any problems.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#answerprecheckoutquery
     */
    answerPreCheckoutQuery(ok: boolean, other?: string | Other<"answerPreCheckoutQuery", "pre_checkout_query_id" | "ok">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.refundStarPayment`. Refunds a successful payment in Telegram Stars.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#refundstarpayment
     */
    refundStarPayment(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.editUserStarSubscription`. Allows the bot to cancel or re-enable extension of a subscription paid in Telegram Stars. Returns True on success.
     *
     * @param telegram_payment_charge_id Telegram payment identifier for the subscription
     * @param is_canceled Pass True to cancel extension of the user subscription; the subscription must be active up to the end of the current subscription period. Pass False to allow the user to re-enable a subscription that was previously canceled by the bot.
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#edituserstarsubscription
     */
    editUserStarSubscription(telegram_payment_charge_id: string, is_canceled: boolean, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.verifyUser`. Verifies a user on behalf of the organization which is represented by the bot. Returns True on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#verifyuser
     */
    verifyUser(other?: Other<"verifyUser">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.verifyChat`. Verifies a chat on behalf of the organization which is represented by the bot. Returns True on success.
     *
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#verifychat
     */
    verifyChat(other?: Other<"verifyChat">, signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.removeUserVerification`. Removes verification from a user who is currently verified on behalf of the organization represented by the bot. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#removeuserverification
     */
    removeUserVerification(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.removeChatVerification`. Removes verification from a chat that is currently verified on behalf of the organization represented by the bot. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#removechatverification
     */
    removeChatVerification(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.readBusinessMessage`. Marks incoming message as read on behalf of a business account. Requires the can_read_messages business bot right. Returns True on success.
     *
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#readbusinessmessage
     */
    readBusinessMessage(signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.setPassportDataErrors`. Informs a user that some of the Telegram Passport elements they provided contains errors. The user will not be able to re-submit their Passport to you until the errors are fixed (the contents of the field for which you returned the error must change). Returns True on success.
     *
     * Use this if the data submitted by the user doesn't satisfy the standards your service requires for any reason. For example, if a birthday date seems invalid, a submitted document is blurry, a scan shows evidence of tampering, etc. Supply some details in the error message to make sure the user knows how to correct the issues.
     *
     * @param errors An array describing the errors
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#setpassportdataerrors
     */
    setPassportDataErrors(errors: readonly PassportElementError[], signal?: AbortSignal): Promise<true>;
    /**
     * Context-aware alias for `api.sendGame`. Use this method to send a game. On success, the sent Message is returned.
     *
     * @param game_short_name Short name of the game, serves as the unique identifier for the game. Set up your games via BotFather.
     * @param other Optional remaining parameters, confer the official reference below
     * @param signal Optional `AbortSignal` to cancel the request
     *
     * **Official reference:** https://core.telegram.org/bots/api#sendgame
     */
    replyWithGame(game_short_name: string, other?: Other<"sendGame", "chat_id" | "game_short_name">, signal?: AbortSignal): Promise<Message.GameMessage>;
}
type HearsContextCore = FilterCore<":text" | ":caption"> & NarrowMatchCore<string | RegExpMatchArray>;
/**
 * Type of the context object that is available inside the handlers for
 * `bot.hears`.
 *
 * This helper type can be used to narrow down context objects the same way how
 * `bot.hears` does it. This allows you to annotate context objects in
 * middleware that is not directly passed to `bot.hears`, hence not inferring
 * the correct type automatically. That way, handlers can be defined in separate
 * files and still have the correct types.
 */
export type HearsContext<C extends Context> = Filter<NarrowMatch<C, string | RegExpMatchArray>, ":text" | ":caption">;
type CommandContextCore = FilterCore<":entities:bot_command"> & NarrowMatchCore<string>;
/**
 * Type of the context object that is available inside the handlers for
 * `bot.command`.
 *
 * This helper type can be used to narrow down context objects the same way how
 * `bot.command` does it. This allows you to annotate context objects in
 * middleware that is not directly passed to `bot.command`, hence not inferring
 * the correct type automatically. That way, handlers can be defined in separate
 * files and still have the correct types.
 */
export type CommandContext<C extends Context> = Filter<NarrowMatch<C, string>, ":entities:bot_command">;
type NarrowMatchCore<T extends Context["match"]> = {
    match: T;
};
type NarrowMatch<C extends Context, T extends C["match"]> = {
    [K in keyof C]: K extends "match" ? (T extends C[K] ? T : never) : C[K];
};
type CallbackQueryContextCore = FilterCore<"callback_query:data">;
/**
 * Type of the context object that is available inside the handlers for
 * `bot.callbackQuery`.
 *
 * This helper type can be used to annotate narrow down context objects the same
 * way `bot.callbackQuery` does it. This allows you to how context objects in
 * middleware that is not directly passed to `bot.callbackQuery`, hence not
 * inferring the correct type automatically. That way, handlers can be defined
 * in separate files and still have the correct types.
 */
export type CallbackQueryContext<C extends Context> = Filter<NarrowMatch<C, string | RegExpMatchArray>, "callback_query:data">;
type GameQueryContextCore = FilterCore<"callback_query:game_short_name">;
/**
 * Type of the context object that is available inside the handlers for
 * `bot.gameQuery`.
 *
 * This helper type can be used to narrow down context objects the same way how
 * `bot.gameQuery` does it. This allows you to annotate context objects in
 * middleware that is not directly passed to `bot.gameQuery`, hence not
 * inferring the correct type automatically. That way, handlers can be defined
 * in separate files and still have the correct types.
 */
export type GameQueryContext<C extends Context> = Filter<NarrowMatch<C, string | RegExpMatchArray>, "callback_query:game_short_name">;
type InlineQueryContextCore = FilterCore<"inline_query">;
/**
 * Type of the context object that is available inside the handlers for
 * `bot.inlineQuery`.
 *
 * This helper type can be used to narrow down context objects the same way how
 * annotate `bot.inlineQuery` does it. This allows you to context objects in
 * middleware that is not directly passed to `bot.inlineQuery`, hence not
 * inferring the correct type automatically. That way, handlers can be defined
 * in separate files and still have the correct types.
 */
export type InlineQueryContext<C extends Context> = Filter<NarrowMatch<C, string | RegExpMatchArray>, "inline_query">;
type ReactionContextCore = FilterCore<"message_reaction">;
/**
 * Type of the context object that is available inside the handlers for
 * `bot.reaction`.
 *
 * This helper type can be used to narrow down context objects the same way how
 * annotate `bot.reaction` does it. This allows you to context objects in
 * middleware that is not directly passed to `bot.reaction`, hence not inferring
 * the correct type automatically. That way, handlers can be defined in separate
 * files and still have the correct types.
 */
export type ReactionContext<C extends Context> = Filter<C, "message_reaction">;
type ChosenInlineResultContextCore = FilterCore<"chosen_inline_result">;
/**
 * Type of the context object that is available inside the handlers for
 * `bot.chosenInlineResult`.
 *
 * This helper type can be used to narrow down context objects the same way how
 * annotate `bot.chosenInlineResult` does it. This allows you to context objects in
 * middleware that is not directly passed to `bot.chosenInlineResult`, hence not
 * inferring the correct type automatically. That way, handlers can be defined
 * in separate files and still have the correct types.
 */
export type ChosenInlineResultContext<C extends Context> = Filter<NarrowMatch<C, string | RegExpMatchArray>, "chosen_inline_result">;
type PreCheckoutQueryContextCore = FilterCore<"pre_checkout_query">;
/**
 * Type of the context object that is available inside the handlers for
 * `bot.preCheckoutQuery`.
 *
 * This helper type can be used to narrow down context objects the same way how
 * annotate `bot.preCheckoutQuery` does it. This allows you to context objects in
 * middleware that is not directly passed to `bot.preCheckoutQuery`, hence not
 * inferring the correct type automatically. That way, handlers can be defined
 * in separate files and still have the correct types.
 */
export type PreCheckoutQueryContext<C extends Context> = Filter<NarrowMatch<C, string | RegExpMatchArray>, "pre_checkout_query">;
type ShippingQueryContextCore = FilterCore<"shipping_query">;
/**
 * Type of the context object that is available inside the handlers for
 * `bot.shippingQuery`.
 *
 * This helper type can be used to narrow down context objects the same way how
 * annotate `bot.shippingQuery` does it. This allows you to context objects in
 * middleware that is not directly passed to `bot.shippingQuery`, hence not
 * inferring the correct type automatically. That way, handlers can be defined
 * in separate files and still have the correct types.
 */
export type ShippingQueryContext<C extends Context> = Filter<NarrowMatch<C, string | RegExpMatchArray>, "shipping_query">;
type ChatTypeContextCore<T extends Chat["type"]> = T extends unknown ? Record<"update", ChatTypeUpdate<T>> & ChatType<T> & Record<"chatId", number> & ChatFrom<T> & ChatTypeRecord<"msg", T> & AliasProps<ChatTypeUpdate<T>> : never;
/**
 * Type of the context object that is available inside the handlers for
 * `bot.chatType`.
 *
 * This helper type can be used to narrow down context objects the same way how
 * `bot.chatType` does it. This allows you to annotate context objects in
 * middleware that is not directly passed to `bot.chatType`, hence not inferring
 * the correct type automatically. That way, handlers can be defined in separate
 * files and still have the correct types.
 */
export type ChatTypeContext<C extends Context, T extends Chat["type"]> = T extends unknown ? C & ChatTypeContextCore<T> : never;
type ChatTypeUpdate<T extends Chat["type"]> = ChatTypeRecord<"message" | "edited_message" | "channel_post" | "edited_channel_post" | "my_chat_member" | "chat_member" | "chat_join_request", T> & Partial<Record<"callback_query", ChatTypeRecord<"message", T>>> & ConstrainUpdatesByChatType<T>;
type ConstrainUpdatesByChatType<T extends Chat["type"]> = Record<[
    T
] extends ["channel"] ? "message" | "edited_message" : "channel_post" | "edited_channel_post", undefined>;
type ChatTypeRecord<K extends string, T extends Chat["type"]> = Partial<Record<K, ChatType<T>>>;
interface ChatType<T extends Chat["type"]> {
    chat: {
        type: T;
    };
}
interface ChatFrom<T extends Chat["type"]> {
    from: [T] extends ["private"] ? {} : unknown;
}
import { AbortSignal } from "./shim.node.js";
export {};
